/**************************************************************************/
/* DESCRIPTION: This file is part of the HILBERT program package for the  */
/*        numerical solution of the Laplace equation with mixed boundary  */
/*        conditions by use of BEM in 2D. It provides some preprocessor   */
/*        constants and functions to                                      */
/*        calculate the distance between a segment and a point and two    */
/*        segments.                                                       */
/**************************************************************************/
/* VERSION: 3.1                                                           */
/**************************************************************************/
/* (C) 2009-2013 HILBERT-Team '09, '10, '12                               */
/* support + bug report:  hilbert@asc.tuwien.ac.at                        */
/**************************************************************************/
#ifndef _GEOMETRY_H_GUARD_
#define _GEOMETRY_H_GUARD_

#include <math.h>
#include <stdio.h>
#include <assert.h>

/************************* distancePointToSegment *************************/
/* INPUT: double p0, p1; double a0, a1; double b0, b1; are the            */
/*        coordinates of three points P, A and B.                         */
/* OUTPUT: double; distance between the point P and the segment [A,B].    */
/* DESCRIPTION: This function treats the segment as a parametrized vector */
/*        where the parameter t varies from 0 to 1. It finds the value    */
/*        of t that minimizes the distance from the point to the line.    */
/*        If t is between 0.0 and 1.0, then the closest point lies on     */
/*        the segment. Otherwise the closest point is one of the segments */
/*        end points.                                                     */
/**************************************************************************/
double distancePointToSegment(double p0, double p1, double a0, double a1,
                                                    double b0, double b1);

/********************************** ccw ***********************************/
/* INPUT: double a0, a1, b0, b1, x0, x1, the coordinates of three points  */
/*        A = (a0 | a1), B = (b0 | b1), X = (x0 | x1).                    */
/* OUTPUT: an integer being either -1, 0 or 1.                            */
/* DESCRIPTION: This function returns 1, in case the angle ABX is in      */
/*        (0,pi), i.e. the segment AB can be transformed into the segment */
/*        AX using a rotation in mathematical positive sense              */
/*        (counter-clock wise). It returns -1 in the other case, ABX in   */
/*        (pi,2pi). In case, A, B, X are collinear, ABX is either 0 or    */
/*        pi. The function returns 0, in case X is in [A,B]. If it's not, */
/*        either -1 is returned, in case that X is closer to A or 1 is    */
/*        returned if X is closer to B.                                   */
/*                                                                        */
/*        This function is called by distanceSegmentToSegment multiple    */
/*        times to check whether two segments intersect.                  */
/**************************************************************************/
int ccw(double a0, double a1, double b0, double b1, double x0, double x1);

/******************************** distance ********************************/
/* INPUT: double a0, a1; double b0, b1; double c0, c1; double d0, d1; are */
/*        coordinates of the points A, B, C, D.                           */
/* OUTPUT: double; distance between the segments [A,B] and [C,D].         */
/* DESCRIPTION: The distance between two segments is either zero if they  */
/*        intersect or the distance from one of the segments' end points  */
/*        to the other line. "distance" first checks if the segments      */
/*        intersect. In case they do, the function returns 0.             */
/*        Otherwise it calculates the distance between each end point and */
/*        the respective other segment and returns the minimum.           */
/**************************************************************************/
double distanceSegmentToSegment(double a0, double a1, double b0, double b1, 
                                double c0, double c1, double d0, double d1);

/********************************** dot ***********************************/
/* INPUT: double u[2] and double v[2] are vectors in R^2                  */
/* OUTPUT: double; the scalar product of the vectors u and v              */
/* DESCRIPTION: This simple and widely used function computes the scalar  */
/*        product of two vectors u and v                                  */
/**************************************************************************/
double dot(double u[2], double v[2]);

#endif
