/**************************************************************************/
/* DESCRIPTION: This file is part of the HILBERT program package for the  */
/*        numerical solution of the Laplace equation with mixed boundary  */
/*        conditions by use of BEM in 2D. It provides functions to        */
/*        calculate the distance between a segment                        */
/*        and a point and between two segments respectively.              */
/*                                                                        */
/*        This file contains only the implementation. For documentation   */
/*        see geometry.h.                                                 */
/**************************************************************************/
/* VERSION: 3.1                                                           */
/**************************************************************************/
/* (C) 2009-2013 HILBERT-Team '09, '10, '12                               */
/* support + bug report:  hilbert@asc.tuwien.ac.at                        */
/**************************************************************************/
#include "geometry.h"

double distancePointToSegment(double p0,double p1, double a0,double a1,
                                                   double b0,double b1) {
  double t = 0.;
  double dab0 = b0 - a0;    /* Distance between A and B.  */
  double dab1 = b1 - a1;
  double r0 = 0.;           /* Distance between [A,B] and P. */
  double r1 = 0.;

  if (fabs(dab0) <= EPS && fabs(dab1) <= EPS) {
    /* Case that the segment is shorter than sqrt(2)*EPS.*/
    r0 = p0 - a0;
    r1 = p1 - a1;
  }
  else {
    t = ((p0-a0)*(b0-a0) + (p1-a1)*(b1-a1)) / 
          ((b0-a0)*(b0-a0) + (b1-a1)*(b1-a1));

    if (t <= EPS) {
        r0 = p0-a0;
        r1 = p1-a1; 
    }
    else if ((1-t) <= EPS) {
        r0 = p0-b0;
        r1 = p1-b1;
    }
    else {
        r0 = p0 - a0 - t*dab0;
        r1 = p1 - a1 - t*dab1;
    }
  }

  return sqrt(r0*r0 + r1*r1);
}

int ccw(double a0, double a1, double b0, double b1, double x0, double x1) {
  double dab0 = 0., dab1 = 0., dax0 = 0., dax1 = 0.;

  assert(a0 != b0 || a1 != b1);

  dab0 = b0 - a0;
  dab1 = b1 - a1;
  dax0 = x0 - a0;
  dax1 = x1 - a1;

  /* dab1 / dab0 < dax1 / dax0 <=> dab0 * dax1 > dax0 * dab1 */
  /* dab0 / dab1 > dax0 / dax1 is equivalent to the fact that the angle
       ABX is in (0,pi). */
  if (dab0 * dax1 > dax0 * dab1) {
    return +1;
  }
  else if (dab0 * dax1 < dax0 * dab1) {
    /* Contrary, this is equivalent to the fact that ABX is in (pi,2pi). */
    return -1;
  }
  else { /* Otherwise, A, B, X are colinear. */
    if (dab0 * dax0 < 0 || dab1 * dax1 < 0) { /* X is closer to A */
      return -1;
    }
    else if (dab0 * dab0 + dab1 * dab1 < dax0 * dax0 + dax1 * dax1) {
      return 1;                               /* X is closer to B */
    }
    else { /* X is in [A,B]. */
      return 0;
    }
  }
}

double distanceSegmentToSegment(double a0, double a1, double b0, double b1,
                                double c0, double c1, double d0, double d1)
{
  double test = 0.;
  double best = sqrt((a0-c0)*(a0-c0)+(a1-c1)*(a1-c1));

  /* The only possibility that two segments intersect: */
  if (ccw(a0, a1, b0, b1, c0, c1) * ccw(a0, a1, b0, b1, d0, d1) <= 0 &&
      ccw(c0, c1, d0, d1, a0, a1) * ccw(c0, c1, d0, d1, b0, b1) <= 0) {
    best = 0.;
  }
  else { /* If segements do not intersect: */
    /* Calculate the distance from A to segment [C,D]: */
    test = distancePointToSegment(a0, a1, c0, c1, d0, d1);
    if ((test-best) <= EPS) {
        best = test;
    }
    /* Calculate the distance from B to segment [C,D]: */
    test = distancePointToSegment(b0, b1, c0, c1, d0, d1);
    if ((test-best) <= EPS) {
        best = test;
    }
    /* Calculate the distance from C to segment [A,B]: */
    test = distancePointToSegment(c0, c1, a0, a1, b0, b1);
    if ((test-best) <= EPS) {
        best = test;
    }
    /* Calculate the distance from D to segment [A,B]: */
    test = distancePointToSegment(d0, d1, a0, a1, b0, b1);
    if ((test-best) <= EPS) {
        best = test;
    }
  }

  return best;
}

double dot(double u[2], double v[2]) {
  return (u[0]*v[0] + u[1]*v[1]);
}

