/**************************************************************************/
/* DESCRIPTION: This file is part of the HILBERT program package for the  */
/*        numerical solution of the Laplace equation with mixed boundary  */
/*        conditions by use of BEM in 2D.                                 */
/*                                                                        */
/*        This file contains the function evaluateW that evaluates the    */
/*        hypersingular integral operator W on any number of evaluation   */
/*        points on the boundary Gamma.                                   */
/*                                                                        */
/*        This file contains only the implementation. For extensive       */
/*        documentation consult the corresponding header-file.            */
/**************************************************************************/
/* VERSION: 3.1                                                           */
/**************************************************************************/
/* (C) 2009-2013 HILBERT-Team '10, '12                                    */
/* support + bug report:  hilbert@asc.tuwien.ac.at                        */
/**************************************************************************/
#include "mex.h"

#include <assert.h>
#include <math.h>

#include "constants.h"
#include "doubleLayerPotential.h"
#include "evaluateW.h"
#include "gaussQuadrature.h"
#include "geometry.h"

void mexFunction(int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[])
{
  const char* function_name = mexFunctionName();
  char error_message[255];
  int nC = 0, nE = 0, nI = 0;
  double* coordinates = NULL;
  double* elements = NULL;
  double* n_x = NULL;
  double* gh = NULL;
  double eta = DEFAULT_ETA;
  double* x = NULL;
  double* Wx = NULL;

  /*Read input data*/
  switch(nrhs)
  {
    case 6:
      eta = mxGetScalar(prhs[5]);
      if (eta < 0)
      {
        sprintf(error_message,
              "In %s, ETA is less than zero (ETA=%f).\n"
              "Please choose a value that is greater than or equal to 0.\n",
              function_name,eta);
        mexErrMsgTxt(error_message);
      }
    case 5:
      coordinates = mxGetPr(prhs[0]);
      nC = mxGetM(prhs[0]);
      elements = mxGetPr(prhs[1]);
      nE = mxGetM(prhs[1]);
      gh = mxGetPr(prhs[2]);
      x = mxGetPr(prhs[3]);
      nI = mxGetM(prhs[3]);
      n_x = mxGetPr(prhs[4]);
      break;
    default:
      sprintf(error_message,
            "Invalid number of arguments: Usage\n"
            "Wphi_x=%s(coordinates,elements,gh,x,n_x[,eta])\n",
            function_name);
      mexErrMsgTxt(error_message);
      break;
  }

  if (nC != mxGetM(prhs[2]) || 1 != mxGetN(prhs[2]))
  {
    sprintf(error_message,
            "In %s, the third parameter (gh) has to be an nCx1 matrix, "
            "where nC denotes the number of vertices, i.e. the number of "
            "rows of the first element.", function_name);
    mexErrMsgTxt(error_message);
  }

  if (nI != mxGetM(prhs[4]))
  {
    sprintf(error_message,
            "In %s, the number of evaluation points (%d)\n"
            "must correspond to the number of normal vectors (%d).\n",
            function_name,nI,(int)mxGetM(prhs[4]));
    mexErrMsgTxt(error_message);
  }

  /* Allocate memory for output data */
  plhs[0] = mxCreateDoubleMatrix(nI,1,mxREAL);
  Wx=mxGetPr(plhs[0]);
  evaluateW(Wx,nC,nE,coordinates,elements,gh,x,nI,n_x,eta);
}

void evaluateW(double* Wx, int nC ,int nE, double* coordinates,
           double* elements, double* gh, double* x,int nI, double* n_x,
           double eta)
{
  int i = 0, j = 0, k = 0;
  double x1,y1,x2,y2,mpx,mpy,vcx,vcy;
  double a[2];
  double b[2];
  double a_dot_b, a_sqabs, b_sqabs;
  double delta, sqrdelta;
  double g_0m1, g_1m1;
  double g_0m2, g_1m2, g_2m2;
  double W_0=0, W_1=0;
  double n_vc[2];
  double dist_x_Ej=0;
  int n_gauss_points=GAUSS_ORDER;
  const double* q_points=getGaussPoints(GAUSS_ORDER);
  const double* q_weights=getGaussWeights(GAUSS_ORDER);

  /*Initialize output vector*/
  for (j = 0; j < nI; ++j)
  {
    Wx[j]=0;
  }

  for(j = 0; j < nI; ++j)
  {
    /* Find the element which contains the current evaluation point.
     * Warn the user if
     *   - No such element is found.
     *   - More than one such element is found.
     *   - The evaluation point is very close to an end point of that
     *      element.
     * Either of these cases results in unreliable results and should
     * be avoided.
     */
    
    int found_element = 0; /* Contains the number of elements containing
                            * the evaluation point. */
    int j_element = 0;     /* Contains the id of the last found element. */
    double x_to_xy1 = 0.;  /* Contains length of vector from the evaluation
                            * point to the start point of the element
                            * containing the evaluation point.
                            */
    double s = 0.;         /* Contains ratio x_to_xy1 / length of element.*/
    double gh_xj = 0.;     /* Contains function value of gh in evaluation
                            * point.
                            */

    for (i = 0; i < nE; ++i)
    {
      double lengthEj = 0.;
      x1=coordinates[(int)elements[i]-1];
      y1=coordinates[(int)elements[i]+nC-1];
      x2=coordinates[(int)elements[i+nE]-1];
      y2=coordinates[(int)elements[i+nE]+nC-1];
      lengthEj = sqrt((x2-x1)*(x2-x1) + (y2-y1)*(y2-y1));
      dist_x_Ej = distancePointToSegment(x[j], x[j+nI], x1, y1, x2, y2);

      if (dist_x_Ej < 5e-2 * EPS) 
      {
        double gh_xy1 = 0.; /* Value of gh in start point of element. */
        double gh_xy2 = 0.; /* Value of gh in end point of element. */

        found_element += 1;
        j_element = i;

        if (sqrt((x1-x[j])*(x1-x[j]) + (y1-x[nI+j])*(y1-x[nI+j])) < EPS
            || sqrt((x2-x[j])*(x2-x[j]) + (y2-x[nI+j])*(y2-x[nI+j])) < EPS)
        {
          fprintf(stderr, "Warning (evaluateW): The evaluation point "
            "(%g,%g) is too close to the end point of an element. The "
            "%d-th value of the return vector is probably wrong!\n",
            x[j], x[nI+j], j);
        }

        x_to_xy1 = sqrt((x[j]-x1)*(x[j]-x1) + (x[nI+j]-y1)*(x[nI+j]-y1));
        assert(x_to_xy1 >= 0. && x_to_xy1 <= lengthEj);

        s = (2. * x_to_xy1/lengthEj) - 1.;
        assert(s < 1. && s > -1.);

        assert((int)elements[i] > 0 && (int)elements[i] <= nC);
        assert((int)elements[nE+i] > 0 && (int)elements[nE+i] <= nC);

        gh_xy1 = gh[(int)elements[i]-1];
        gh_xy2 = gh[(int)elements[nE+i]-1];
        gh_xj = gh_xy1 + (x_to_xy1 / lengthEj) * (gh_xy2-gh_xy1);

        break;
      }
    }

    if (found_element == 0)
    {
      fprintf(stderr, "Warning (evaluateW): The evaluation point "
        "(%g,%g) is not even close to an element! The %d-th entry of "
        "the return vector is set to NaN!\n", x[j], x[nI+j], j);
      Wx[j] = NAN;
      continue;
    }

    if (found_element > 1)
    {
      fprintf(stderr, "Warning (evaluateW): The evaluation point "
        "(%g,%g) is very close to %d elements. This either means that "
        "the evaluation point is very close to the end point of an element "
        "or that the input mesh is invalid. The %d-th entry of the return "
        "vector is set to NaN!\n", x[j], x[nI+j], found_element, j);
      Wx[j] = NAN;
      continue;
    }

    /*Computation of Wx*/
    for(i = 0; i < nE; ++i)
    {
      x1=coordinates[(int)elements[i]-1];
      y1=coordinates[(int)elements[i]+nC-1];
      x2=coordinates[(int)elements[i+nE]-1];
      y2=coordinates[(int)elements[i+nE]+nC-1];

      mpx = 0.5 * (x1+x2);
      mpy = 0.5 * (y1+y2);

      vcx = x2 - x1;
      vcy = y2 - y1;

      a[0] = -0.5*vcx;
      a[1] = -0.5*vcy;

      b[0] = x[j]    - mpx;
      b[1] = x[j+nI] - mpy;

      a_dot_b  = a[0]*b[0] + a[1]*b[1];
      a_sqabs  = a[0]*a[0] + a[1]*a[1];
      b_sqabs  = b[0]*b[0] + b[1]*b[1];
      delta    = a_sqabs*b_sqabs - a_dot_b*a_dot_b;
      sqrdelta = 2 * sqrt(delta);

      g_0m1 = 0;
      g_1m1 = 0;
      g_0m2 = 0;
      g_1m2 = 0;
      g_2m2 = 0;
      W_0 = 0;
      W_1 = 0;

      /*Determine the normal vector*/
      n_vc[0] =  0.5*vcy / sqrt(a_sqabs);
      n_vc[1] = -0.5*vcx / sqrt(a_sqabs);

      /*Compute distance x to Ej*/
      dist_x_Ej = distancePointToSegment(x[j], x[j+nI], x1, y1, x2, y2);

      /*Check whether the evaluation point is on the current element.*/
      if (dist_x_Ej < 5e-2 * EPS) 
      {
       /*Compute p.v. of a locally regularized integral*/  
        double lengthEj = sqrt(4.0*a_sqabs);
        double alpha = (gh[(int)elements[i+nE]-1]-gh[(int)elements[i]-1])
                            / 2.;
        assert(j_element == i);

        Wx[j] += 1.0 / (lengthEj*M_PI) * (2.0*gh_xj/(1.0-s*s) - 
                             alpha * log(fabs((s-1.0)/(s+1.0))) );
      }
      else
      {
        /*Check if pair (x,T_j) is admissible*/
        if ((sqrt(4*a_sqabs) > eta*dist_x_Ej) || eta==-1)
        {
          /*Compute the integrals analytically*/
          /*Compute g_0^(-1)*/
          g_0m1 = dlp(0,a,b);
          g_1m1 = dlp(1,a,b);
  
          /*Compute g_0^(-2)*/
	  if (sqrdelta*sqrdelta > 4.0*EPS * a_sqabs * b_sqabs)
          {
            g_0m2=(2*(a_sqabs+a_dot_b)/(a_sqabs+2*a_dot_b+b_sqabs)
                   -2*(-a_sqabs+a_dot_b)/(a_sqabs-2*a_dot_b+b_sqabs)
                   +2*a_sqabs*g_0m1)/(sqrdelta*sqrdelta);
          }
          else
          {
            g_0m2=2*(a_sqabs+3*b_sqabs)/(3*(b_sqabs-a_sqabs)
                                     *(b_sqabs-a_sqabs)*(b_sqabs-a_sqabs));
          }
  
          /*Compute g_1^(-2)*/
          g_1m2=(-1/(a_sqabs+2*a_dot_b+b_sqabs)+
                  1/(a_sqabs-2*a_dot_b+b_sqabs)
                  -2*a_dot_b*g_0m2)/(2*a_sqabs);

          /*Compute g_2^(-2)*/
          g_2m2=(g_0m1-2*a_dot_b*g_1m2-b_sqabs*g_0m2)/a_sqabs;
  
          /*Compute W_0*/
          W_0=
            ((n_vc[0]*n_x[j]+n_vc[1]*n_x[j+nI])*g_0m1-
              2*((a[0]*n_x[j]+a[1]*n_x[j+nI])
            *(n_vc[0]*a[0]+n_vc[1]*a[1])*g_1m2+
              ((b[0]*n_x[j]+b[1]*n_x[j+nI])*(n_vc[0]
            *a[0]+n_vc[1]*a[1])+(a[0]*n_x[j]+
              a[1]*n_x[j+nI])*(n_vc[0]*b[0]+n_vc[1]*b[1]))
            *g_1m2+(b[0]*n_x[j]+b[1]*n_x[j+nI])*
              (n_vc[0]*b[0]+n_vc[1]*b[1])*g_0m2))/M_PI;
  
          /*Compute W_1*/
          W_1=
            ((n_vc[0]*n_x[j]+n_vc[1]*n_x[j+nI])*g_1m1-
              2*((a[0]*n_x[j]+a[1]*n_x[j+nI])
            *(n_vc[0]*a[0]+n_vc[1]*a[1])*g_2m2+
              ((b[0]*n_x[j]+b[1]*n_x[j+nI])*(n_vc[0]
            *a[0]+n_vc[1]*a[1])+(a[0]*n_x[j]+
              a[1]*n_x[j+nI])*(n_vc[0]*b[0]+n_vc[1]*b[1]))
            *g_2m2+(b[0]*n_x[j]+b[1]*n_x[j+nI])*
              (n_vc[0]*b[0]+n_vc[1]*b[1])*g_1m2))/M_PI;
        }
        else
        {
          /*Compute the integrals via quadrature rule*/
          W_0=0;
          W_1=0;
          for(k=0;k<n_gauss_points;++k){
            W_0 += q_weights[k]*((n_x[j]*n_vc[0]+n_x[j+nI]*n_vc[1])/
              (a_sqabs*q_points[k]*q_points[k]+
                2*a_dot_b*q_points[k]+b_sqabs)
              -2*((n_x[j]*(a[0]*q_points[k]+b[0])+
                n_x[j+nI]*(a[1]*q_points[k]+b[1]))
                  *(n_vc[0]*(a[0]*q_points[k]+b[0])+
                  n_vc[1]*(a[1]*q_points[k]+b[1])))/
              ((a_sqabs*q_points[k]*q_points[k]+
                2*a_dot_b*q_points[k]+b_sqabs)*
               (a_sqabs*q_points[k]*q_points[k]+
                2*a_dot_b*q_points[k]+b_sqabs)))/M_PI;
  
            W_1 += q_weights[k]*(q_points[k]*
                    (n_x[j]*n_vc[0]+n_x[j+nI]*n_vc[1])/
                  (a_sqabs*q_points[k]*q_points[k]+
                    2*a_dot_b*q_points[k]+b_sqabs)
                  -2*(q_points[k]*(n_x[j]*(a[0]*q_points[k]+b[0])
                    +n_x[j+nI]*(a[1]*q_points[k]+b[1]))
                  *(n_vc[0]*(a[0]*q_points[k]+b[0])+n_vc[1]
                    *(a[1]*q_points[k]+b[1])))/
                  ((a_sqabs*q_points[k]*q_points[k]+
                    2*a_dot_b*q_points[k]+b_sqabs)*
                  (a_sqabs*q_points[k]*q_points[k]+
                    2*a_dot_b*q_points[k]+b_sqabs)))/M_PI;
          }
        }
        /*Putting the integrals together to obtain W(phi)(x)*/
        Wx[j]=Wx[j]+0.25*sqrt(a_sqabs)*W_1*
                  (gh[(int)elements[i]-1]-gh[(int)elements[i+nE]-1])-
                  0.25*sqrt(a_sqabs)*W_0*(gh[(int)elements[i+nE]-1]+
                  gh[(int)elements[i]-1] );
      }
    }
  }
}

