/**************************************************************************/
/* DESCRIPTION: This file is part of the HILBERT program package for the  */
/*        numerical solution of the Laplace equation with mixed boundary  */
/*        conditions by use of BEM in 2D.                                 */
/*                                                                        */
/*        This file contains the function evaluateV that evaluates the    */
/*        single layer potential operator V tilde on any number of        */
/*        evaluation points within the domain Omega.                      */
/**************************************************************************/
/* VERSION: 3.1                                                           */
/**************************************************************************/
/* (C) 2009-2013 HILBERT-Team '09, '12                                    */
/* support + bug report:  hilbert@asc.tuwien.ac.at                        */
/**************************************************************************/
#ifndef _EVALUATEV_H_GUARD_
#define _EVALUATEV_H_GUARD_

#include "singleLayerPotential.h"
#include "geometry.h"
#include "gaussQuadrature.h"

/****************************** mexFunction *******************************/
/* USAGE: evaluateV(coordinates, elements, phih, x [,eta])                */
/* INPUT: coordinates and elements describe the triangulation of the      */
/*        boundary. phi is a vector that contains the values of phih on   */
/*        the points that are given by the coordinates-matrix, i.e. there */
/*        holds                                                           */
/*                                                                        */
/*        Phi(coordinates(i,:)) = phi(i).                                 */
/*                                                                        */
/*        x is an (M x 2)-matrix that contains the points, V should be    */
/*        evaluated on.                                                   */
/* OUTPUT: Returns an M-dimensional vector where M is the number of       */
/*        evaluation points. The i-th entry of that vector is given by    */
/*                                                                        */
/*        Vtilde Phi(z(i,:)).                                             */
/*                                                                        */
/* DESCRIPTION: This function is a MEX-wrapper around evaluateV.          */
/**************************************************************************/
void mexFunction(int nlhs, mxArray* plhs[],
                  int nrhs, const mxArray* prhs[]);

/******************************* evaluateV ********************************/
/* INPUT: Vphi_z is an nz-dimensional vector, nc is the number of rows of */
/*        coordinates, ne is the number of rows of elements, nz is the    */
/*        number of evaluation points, i.e. the number of rows of z.      */
/*        coordinates and elements are matrices describing the            */
/*        triangulation. phi is an nc-dimensional vector such that phi(i) */
/*        is the function value of Phi on the i-th coordinate             */
/*        coordinates(i,:). z is an (nz x 2) matrix that contains the     */
/*        evaluation points.                                              */
/* OUTPUT: This function modifies the array Vphi_z.                       */
/* DESCRIPTION: evaluateV evaluates the single layer potential operator V */
/*        tilde on a number of evaluation points.                         */
/**************************************************************************/
void evaluateV(double* Vphi_x, int nC, int nE, int nP,
                const double* coordinates, const double* elements,
                const double* phih, const double* x, double eta);

#endif

