/**************************************************************************/
/* DESCRIPTION: This file is part of the HILBERT program package for the  */
/*        numerical solution of the Laplace equation with mixed boundary  */
/*        conditions by use of BEM in 2D.                                 */
/*                                                                        */
/*        This file contains the function evaluateV that evaluates the    */
/*        single layer potential operator V tilde on any number of        */
/*        evaluation points within the domain Omega.                      */
/*                                                                        */
/*        This file contains only the implementation. For extensive       */
/*        documentation consult the corresponding header-file.            */
/**************************************************************************/
/* VERSION: 3.1                                                           */
/**************************************************************************/
/* (C) 2009-2013 HILBERT-Team '09, '12                                    */
/* support + bug report:  hilbert@asc.tuwien.ac.at                        */
/**************************************************************************/
#include "mex.h"

#include "constants.h"
#include "evaluateV.h"

void mexFunction(int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[])
{
  const char* functionName = mexFunctionName();
  char errorMessage[255];
  int nC = 0, nE = 0, nX = 0;
  double eta = DEFAULT_ETA;
  const double *coordinates = NULL, *elements = NULL,
                *phih = NULL, *x = NULL;
  double* Vphi_x = NULL;

  if (nlhs != 1) {
    sprintf(errorMessage,
      "Invalid number of output arguments. Usage:\n"
      "  Vchi_z = %s(coordinates,elements,chi,z)\n",
      functionName);
    mexErrMsgTxt(errorMessage);
  }

  switch (nrhs) {
    case 5:
      eta         = extract_scalar_or(prhs[4], DEFAULT_ETA);
    case 4:
      coordinates = (const double*) mxGetPr(prhs[0]);
      elements    = (const double*) mxGetPr(prhs[1]);
      phih        = (const double*) mxGetPr(prhs[2]);
      x           = (const double*) mxGetPr(prhs[3]);
      nC          = mxGetM(prhs[0]);
      nE          = mxGetM(prhs[1]);
      nX          = mxGetM(prhs[3]);
      break;
    default:
      sprintf(errorMessage,
        "Invalid number of input arguments. Usage:\n"
        "  Vchi_z = %s(coordinates,elements,x,z[,eta])\n",
        functionName);
      mexErrMsgTxt(errorMessage);
  }

  if (eta < 0) {
    sprintf(errorMessage,
      "In %s, ETA is less than zero (ETA=%f).\n"
      "Please choose a value that is greater than or equal to 0.\n",
      functionName, eta);
    mexErrMsgTxt(errorMessage);
  }

  plhs[0] = mxCreateDoubleMatrix(nX, 1, mxREAL);
  Vphi_x  = mxGetPr(plhs[0]);

  evaluateV(Vphi_x, nC, nE, nX, coordinates, elements, phih, x, eta);
}

void evaluateV(double* Vphi_x, int nC, int nE, int nX,
                const double* coordinates, const double* elements,
                const double* phih, const double* x, double eta)
{
  int i = 0, j = 0;
  double* mEj = malloc(sizeof(double) * nE * 2);
  double* dEj = malloc(sizeof(double) * nE * 2);
  double* lengthEj = malloc(sizeof(double) * nE);
  const double* gaussPoint  = getGaussPoints(GAUSS_ORDER);
  const double* gaussWeight = getGaussWeights(GAUSS_ORDER);

  /* For each boundary element Ej = [a,b], we calculate (a+b)/2
   * and (b-a)/2, because we need these vectors multiple times below.
   * Compared to the vectors in coordinates and elements, these vectors
   * are stored in row major order or in other words, we store the
   * transposed vectors ((a+b)/2)^T and ((b-a)/2)^T.
   */
  for (j = 0; j < nE; ++j) {
    int aidx = 0, bidx = 0;
    double a[2], b[2];

    aidx = (int) elements[j]    - 1;    /* Ej = [a,b] */
    bidx = (int) elements[nE+j] - 1;
    a[0] = coordinates[aidx]; a[1] = coordinates[nC+aidx];
    b[0] = coordinates[bidx]; b[1] = coordinates[nC+bidx];

    mEj[2*j]    = 0.5 * (a[0] + b[0]); /* mEj = (a+b)/2 */
    mEj[2*j+1]  = 0.5 * (a[1] + b[1]);

    dEj[2*j]    = 0.5 * (b[0] - a[0]); /* dEj = (b-a)/2 */
    dEj[2*j+1]  = 0.5 * (b[1] - a[1]);

    lengthEj[j] = sqrt((b[0]-a[0])*(b[0]-a[0]) + (b[1]-a[1])*(b[1]-a[1]));
  }

  for (i = 0; i < nX; ++i) {
    Vphi_x[i] = 0.;
    for (j = 0; j < nE; ++j) {
      int aidx = 0, bidx = 0;
      double a[2], b[2];
      double dist_x_Ej = 0.;

      aidx = (int) elements[j]    - 1;    /* Ej = [a,b] */
      bidx = (int) elements[nE+j] - 1;
      a[0] = coordinates[aidx]; a[1] = coordinates[nC+aidx];
      b[0] = coordinates[bidx]; b[1] = coordinates[nC+bidx];

      dist_x_Ej = distancePointToSegment(x[i], x[nX+i], a[0], a[1],
                                          b[0], b[1]);

      if (lengthEj[j] <= eta * dist_x_Ej) {
        int k = 0;
        double sum = 0.;
        for (k = 0; k < GAUSS_ORDER; ++k) {
          double s0 = mEj[2*j]   + gaussPoint[k]*dEj[2*j];
          double s1 = mEj[2*j+1] + gaussPoint[k]*dEj[2*j+1];
          sum += phih[j] * gaussWeight[k] * lengthEj[j]
                  * log((x[i]-s0)*(x[i]-s0)+(x[nX+i]-s1)*(x[nX+i]-s1));
        }
        Vphi_x[i] -= sum;
      }
      else {
        double x_minus_mEj[2];

        x_minus_mEj[0] = x[i]    - mEj[2*j];
        x_minus_mEj[1] = x[nX+i] - mEj[2*j+1];

        Vphi_x[i] -= phih[j] * lengthEj[j] *
                        slp(0, &(dEj[2*j]), x_minus_mEj);
      }
    }
    Vphi_x[i] /= 8.*M_PI;
  }

  free(mEj);
  free(dEj);
  free(lengthEj);
}

