/**************************************************************************/
/* DESCRIPTION: This file is part of the HILBERT program package for the  */
/*        numerical solution of the Laplace equation with mixed boundary  */
/*        conditions by use of BEM in 2D.                                 */
/**************************************************************************/
/* VERSION: 3.1                                                           */
/**************************************************************************/
/* (C) 2009-2013 HILBERT-Team '11                                         */
/* support + bug report:  hilbert@asc.tuwien.ac.at                        */
/**************************************************************************/
#include "mex.h"
#include "constants.h"
#include "evaluateN.h"

void mexFunction(int nlhs, mxArray** plhs, int nrhs, const mxArray** prhs)
{
  const char* function_name = mexFunctionName();
  char error_message[150];
  int nVert=0, nVol=0, np=0;
  const double* vertices=NULL;
  const double* volumes=NULL;
  const double* x=NULL;
  const double* f=NULL;
  
  double* s=NULL;

  if(nlhs!=1){
  sprintf(error_message,"Invalid number of output arguments. Use\n"
          "  S = %s(vertices,volumes,f,x)\n",
          function_name);
  mexErrMsgTxt(error_message); 
  }

  /* Read input data */
  switch (nrhs) {
    case 4:
      vertices = (const double*) mxGetPr(prhs[0]);
      nVert = mxGetM(prhs[0]);
      volumes = (const double*) mxGetPr(prhs[1]);
      nVol = mxGetM(prhs[1]);
      f = (const double*) mxGetPr(prhs[2]);
      x = (const double*) mxGetPr(prhs[3]);
      np = mxGetM(prhs[3]); 
      break;
    default:
      sprintf(error_message,
	  "Invalid number of input arguments (%d). Use\n"
          "  S = %s(vertices,volumes,f,x)\n",
	  nrhs, function_name);
      mexErrMsgTxt(error_message);
      return;
  }

  /* allocate memory for output data */
  plhs[0] = mxCreateDoubleMatrix(np,1,mxREAL);
  s = mxGetPr(plhs[0]);

  /* evaluate Newton potential at given points */
  evaluateN(s,vertices,volumes,f,x,nVert,nVol,np);
}

void evaluateN(double* s, const double* vertices, const double* volumes,
	       const double* f, const double* x, int nVert, int nVol, int np) {

  int m=0,n=0;
  double x_m[2];
  double z1[2],z2[2],z3[2];
    
  for(m=0; m<np; ++m) { /* run over evaluation points */
    s[m] = 0.0;
    x_m[0] = x[m];
    x_m[1] = x[m+np];
    for(n=0; n<nVol; ++n) { /* run over volumes */
      z1[0] = vertices[(int)volumes[n]-1];
      z1[1] = vertices[(int)volumes[n]-1+nVert];
      z2[0] = vertices[(int)volumes[n+nVol]-1];
      z2[1] = vertices[(int)volumes[n+nVol]-1+nVert];
      z3[0] = vertices[(int)volumes[n+2*nVol]-1];
      z3[1] = vertices[(int)volumes[n+2*nVol]-1+nVert];
        
      s[m] += f[n]*newtonPotential(z1,z2,z3,x_m);
    }
    s[m] /= (-2*M_PI);
  }
}

