/**************************************************************************/
/* DESCRIPTION: This file is part of the HILBERT program package for the  */
/*        numerical solution of the Laplace equation with mixed boundary  */
/*        conditions by use of BEM in 2D.                                 */
/*                                                                        */
/*        This file contains the function evaluateKadj that evaluates     */
/*        the adjoint double layer potential operator K* on any number of */
/*        evaluation points on the boundary Gamma.                        */
/*                                                                        */
/*        This file contains only the implementation. For extensive       */
/*        documentation consult the corresponding header-file.            */
/**************************************************************************/
/* VERSION: 3.1                                                           */
/**************************************************************************/
/* (C) 2009-2013 HILBERT-Team '10, '12                                    */
/* support + bug report:  hilbert@asc.tuwien.ac.at                        */
/**************************************************************************/
#include "mex.h"

#include "constants.h"
#include "doubleLayerPotential.h"
#include "evaluateKadj.h"
#include "gaussQuadrature.h"
#include "geometry.h"

void mexFunction(int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[])
{
  const char* function_name = mexFunctionName();
  char error_message[255];
  int nC=0;
  int nE=0;
  double* n_x=NULL;
  int nI=0;
  double* elements=NULL;
  double* coordinates=NULL;
  double* gh=NULL;
  double eta=DEFAULT_ETA;
  double* x=NULL;
  double* Kx=NULL;

  /*Read input data*/
  switch(nrhs)
  {
    case 6:
      eta = extract_scalar_or(prhs[5], DEFAULT_ETA);
    case 5:
      coordinates=mxGetPr(prhs[0]);
      nC=mxGetM(prhs[0]);
      elements=mxGetPr(prhs[1]);
      nE = mxGetM(prhs[1]);
      gh=mxGetPr(prhs[2]);
      x=mxGetPr(prhs[3]);
      nI=mxGetM(prhs[3]);
      n_x=mxGetPr(prhs[4]);
      break;
    default:
     sprintf(error_message,
              "Invalid number of arguments: Usage\n"
              "Vphi_x=%s(coordinates,elements,gh,x,n_x[,eta])\n",
              function_name);
      mexErrMsgTxt(error_message);
      break;
  }

  if (eta < 0)
  {
    sprintf(error_message,
            "In %s, ETA is less than zero (ETA=%f).\n"
            "Please choose a value that is greater than or equal to 0.\n",
            function_name,eta);
    mexErrMsgTxt(error_message);
  }

  if (nI != mxGetM(prhs[4]))
  {
    sprintf(error_message,
            "In %s, the number of evaluation points (%d)\n"
            "must correspond to the number of normal vectors (%d).\n",
            function_name,nI,(int)mxGetM(prhs[4]));
  }

  /* Allocate memory for output data */
  plhs[0] = mxCreateDoubleMatrix(nI,1,mxREAL);
  Kx = mxGetPr(plhs[0]);
  evaluateKadj(Kx,nC,nE,coordinates,elements,gh,x,nI,n_x,eta);
}

void evaluateKadj(double* Kx, int nC ,int nE, double* coordinates,
  double* elements, double* gh, double* x,int nI, double* n_x, double eta)
{
  int i=0, j=0, k=0;
  double x1,y1,x2,y2,mpx,mpy,vcx,vcy;
  double a[2];
  double b[2];
  double a_dot_b, a_sqabs, b_sqabs;
  double sqrdelta;
  double g_0m1, g_1m1, g_2m1;
  double K_0=0;
  double n_vc[2];
  double dist_x_Ej=0;
  int n_gauss_points=GAUSS_ORDER;
  const double* q_points  = getGaussPoints(GAUSS_ORDER);
  const double* q_weights = getGaussWeights(GAUSS_ORDER);

  /* Initialize output vector */
  for (i=0;i<nI;++i)
  {
    Kx[i]=0;
  }

  for (j=0;j<nI;++j)
  {
    /*Computation of Kx*/
    for (i=0;i<nE;++i)
    {
      /* The current element's end points are stored in (x1,y1), (x2,y2),
       * the mid point is stored in (mpx,mpy) and the vector from (x1,y1) to
       * (x2,y2) in vcx.
       */
      x1 = coordinates[(int)elements[i]-1];
      y1 = coordinates[(int)elements[i]+nC-1];

      x2 = coordinates[(int)elements[i+nE]-1];
      y2 = coordinates[(int)elements[i+nE]+nC-1];

      mpx = 0.5*(x1+x2);
      mpy = 0.5*(y1+y2);

      vcx = x2-x1;
      vcy = y2-y1;

      a[0] = -0.5 * vcx;
      a[1] = -0.5 * vcy;

      b[0] = x[j] - mpx;
      b[1] = x[j+nI] - mpy;

      a_dot_b = a[0]*b[0] + a[1]*b[1];
      a_sqabs = a[0]*a[0] + a[1]*a[1];
      b_sqabs = b[0]*b[0] + b[1]*b[1];

      sqrdelta = sqrt(a_sqabs*b_sqabs - a_dot_b*a_dot_b);

      g_0m1 = 0;
      g_1m1 = 0;
      g_2m1 = 0;
      K_0 = 0;

      /*Determine the normal vector*/
      n_vc[0] =  (0.5 * vcy) / sqrt(a_sqabs);
      n_vc[1] = -(0.5 * vcx) / sqrt(a_sqabs);

      /*Check if element is admissible*/
      dist_x_Ej = distancePointToSegment(x[j], x[j+nI], x1, y1, x2, y2);

      if (sqrt(2*a_sqabs) > eta*dist_x_Ej) /* x and Ej are inadmissable. */
      {
        /*Compute the integrals analytically*/

        /* It is not safe to call dlp if a, b have the same length and
         * are linearly dependent. But fortunately, this is only the case
         * if the whole integral is 0.
         */

        if (sqrdelta > EPS * sqrt(a_sqabs * b_sqabs))
        {
          g_0m1 = dlp(0,a,b); /*Compute g_0^(-1)*/
          g_1m1 = dlp(1,a,b); /*Compute g_1^(-1)*/

          /*Compute K_0*/
          K_0 = ((n_x[j]*a[0] + n_x[j+nI]*a[1]) * g_1m1 +
                  (n_x[j]*b[0] + n_x[j+nI]*b[1]) * g_0m1);
          K_0 /= M_PI;
        }
        else
        {
          K_0 = 0.;
        }
      }
      else
      {
        /*Compute the integrals via quadrature rule*/
        K_0=0;

        for (k=0; k<n_gauss_points; ++k)
        {
          double contrib = 
            (n_x[j] * (a[0]*q_points[k] + b[0]) +
             n_x[j+nI] * (a[1]*q_points[k] + b[1]))
                / ((a_sqabs*q_points[k]*q_points[k] +
                    2. * a_dot_b * q_points[k] +
                    b_sqabs)
                    * M_PI);
          K_0 += q_weights[k] * contrib;
        }
      }

      Kx[j] -= 0.5 * gh[i] * sqrt(a_sqabs) * K_0;
    }
  }
}

