/**************************************************************************/
/* DESCRIPTION: This file is part of the HILBERT program package for the  */
/*        numerical solution of the Laplace equation with mixed boundary  */
/*        conditions by use of BEM in 2D.                                 */
/*                                                                        */
/*        This file contains the function evaluateK that evaluates the    */
/*        double layer potential operator K tilde on any number of        */
/*        evaluation points within the domain Omega.                      */
/*                                                                        */
/*        This file contains only the implementation. For extensive       */
/*        documentation consult the corresponding header-file.            */
/**************************************************************************/
/* VERSION: 3.1                                                           */
/**************************************************************************/
/* (C) 2009-2013 HILBERT-Team '09, '12                                    */
/* support + bug report:  hilbert@asc.tuwien.ac.at                        */
/**************************************************************************/
#include "mex.h"

#include "constants.h"
#include "evaluateK.h"

void mexFunction(int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[])
{
  const char* functionName = mexFunctionName();
  char errorMessage[255];
  int nC = 0, nE = 0, nX = 0;
  double eta = DEFAULT_ETA;
  const double *coordinates = NULL, *elements = NULL, *gh = NULL, *x = NULL;
  double* gx = NULL;

  if (nlhs != 1) {
    sprintf(errorMessage,
      "Invalid number of output arguments. Usage:\n"
      "  gx = %s(coordinates,elements,gh,x)\n",
      functionName);
    mexErrMsgTxt(errorMessage);
  }

  switch (nrhs) {
    case 5:
      eta         = extract_scalar_or(prhs[4], DEFAULT_ETA);
    case 4:
      coordinates = (const double*) mxGetPr(prhs[0]);
      elements    = (const double*) mxGetPr(prhs[1]);
      gh          = (const double*) mxGetPr(prhs[2]);
      x           = (const double*) mxGetPr(prhs[3]);
      nC          = mxGetM(prhs[0]);
      nE          = mxGetM(prhs[1]);
      nX          = mxGetM(prhs[3]);
      break;
    default:
      sprintf(errorMessage,
        "Invalid number of input arguments. Usage:\n"
        "  Vchi_z = %s(coordinates,elements,x,z[,eta])\n",
        functionName);
      mexErrMsgTxt(errorMessage);
  }

  if (eta < 0) {
    sprintf(errorMessage,
      "In %s, ETA is less than zero (ETA=%f).\n"
      "Please choose a value that is greater than or equal to 0.\n",
      functionName, eta);
    mexErrMsgTxt(errorMessage);
  }

  plhs[0] = mxCreateDoubleMatrix(nX, 1, mxREAL);
  gx      = mxGetPr(plhs[0]);

  evaluateK(gx, nC, nE, nX, coordinates, elements, gh, x, eta);
}

void evaluateK(double* gx, int nC, int nE, int nX,
                const double* coordinates, const double* elements,
                const double* gh, const double* x, double eta)
{
  int i = 0, j = 0;
  double *nEj = NULL, *mEj = NULL, *dEj = NULL, *lengthEj = NULL;
  const double* gaussPoint  = getGaussPoints(GAUSS_ORDER);
  const double* gaussWeight = getGaussWeights(GAUSS_ORDER);

  nEj = (double*) malloc(sizeof(double) * 2 * nE);
  mEj = (double*) malloc(sizeof(double) * 2 * nE);
  dEj = (double*) malloc(sizeof(double) * 2 * nE);
  lengthEj = (double*) malloc(sizeof(double) * nE);

  for (j = 0; j < nE; ++j) {
    int aidx = 0, bidx = 0;
    double a[2], b[2];

    aidx = ((int) elements[j])    - 1;
    bidx = ((int) elements[nE+j]) - 1;

    a[0] = coordinates[aidx]; a[1] = coordinates[nC+aidx];
    b[0] = coordinates[bidx]; b[1] = coordinates[nC+bidx];

    mEj[2*j]    = 0.5 * (a[0] + b[0]);
    mEj[2*j+1]  = 0.5 * (a[1] + b[1]);

    dEj[2*j]    = 0.5 * (b[0] - a[0]);
    dEj[2*j+1]  = 0.5 * (b[1] - a[1]);

    lengthEj[j] = sqrt((b[0]-a[0])*(b[0]-a[0]) + (b[1]-a[1])*(b[1]-a[1]));

    nEj[2*j]    = (b[1]-a[1])/lengthEj[j];
    nEj[2*j+1]  = (a[0]-b[0])/lengthEj[j];
  }

  for (i = 0; i < nX; ++i) {
    gx[i] = 0.;
    for (j = 0; j < nE; ++j) {
      int aidx = 0, bidx = 0;
      double a[2], b[2];
      double dist_x_Ej = 0.;

      aidx = (int) elements[j]    - 1;
      bidx = (int) elements[nE+j] - 1;
      a[0] = coordinates[aidx]; a[1] = coordinates[nC+aidx];
      b[0] = coordinates[bidx]; b[1] = coordinates[nC+bidx];

      dist_x_Ej = distancePointToSegment(x[i], x[nX+i], a[0], a[1],
                                          b[0], b[1]);

      if (lengthEj[j] <= dist_x_Ej * eta) {
        int k = 0;
        double sum = 0.;

        for (k = 0; k < GAUSS_ORDER; ++k) {
          double s_minus_xi[2];
          double s_minus_xi_dot_nEj = 0.;
          double s0 = mEj[2*j]   + gaussPoint[k]*dEj[2*j];
          double s1 = mEj[2*j+1] + gaussPoint[k]*dEj[2*j+1];

          s_minus_xi[0] = s0-x[i]; s_minus_xi[1] = s1-x[nX+i];
          s_minus_xi_dot_nEj = s_minus_xi[0]*nEj[2*j]
                              + s_minus_xi[1]*nEj[2*j+1];

          sum += gaussWeight[k] * s_minus_xi_dot_nEj
                  * (gh[aidx] + ((1+gaussPoint[k])*(gh[bidx]-gh[aidx])/2))
                  / (s_minus_xi[0]*s_minus_xi[0] +
                      s_minus_xi[1]*s_minus_xi[1]);
        }
        gx[i] -= lengthEj[j] * sum;
      }
      else {
        double mEj_minus_xi[2];
        double mEj_minus_xi_dot_nEj = 0.;
        mEj_minus_xi[0] = mEj[2*j]-x[i];
        mEj_minus_xi[1] = mEj[2*j+1]-x[nX+i];
        mEj_minus_xi_dot_nEj = mEj_minus_xi[0]*nEj[2*j]
                              + mEj_minus_xi[1]*nEj[2*j+1];

        if ( fabs(mEj_minus_xi_dot_nEj) > EPS ) {
          double commonFactor = mEj_minus_xi_dot_nEj * lengthEj[j];
          double prod1 = commonFactor *
                          dlp(0, &(dEj[2*j]), mEj_minus_xi) / 2.;
          double prod2 = commonFactor *
                          dlp(1, &(dEj[2*j]), mEj_minus_xi) / 2.;
          gx[i] -= (((gh[aidx] * prod1 + gh[bidx] * prod1)
                    + gh[bidx] * prod2) - gh[aidx] * prod2);
        }
      }
    }
    gx[i] /= 4. * M_PI;
  }

  free(nEj);
  free(mEj);
  free(dEj);
  free(lengthEj);
}

