/**************************************************************************/
/* DESCRIPTION: This file is part of the HILBERT program package for the  */
/*        numerical solution of the Laplace equation with mixed boundary  */
/*        conditions by use of BEM in 2D. It provides functions to        */
/*        compute the Galerkin-Matrix K given by                          */
/*                                                                        */
/*        K_ij = -1/(2pi)\int_{T_i}\int_{supp phi_j} <y-x,n>/|y-x|^2      */
/*        phi_j(y)ds_y ds_x.                                              */
/*                                                                        */
/**************************************************************************/
/* VERSION: 3.1                                                           */
/**************************************************************************/
/* (C) 2009-2013 HILBERT-Team '09, '10, '12                               */
/* support + bug report:  hilbert@asc.tuwien.ac.at                        */
/**************************************************************************/
#ifndef _DOUBLELAYERPOTENTIAL_H_GUARD
#define _DOUBLELAYERPOTENTIAL_H_GUARD

/********************************** dlp ***********************************/
/* INPUT: k: power of t. p,q: generated by direction vectors of the       */
/*        panels Ti,Tj.                                                   */
/* OUTPUT: result of the integral \int_{-1}^1 t^k/|tp+q|^2 dt             */
/* DESCRIPTION: computes the double-layer potential type integral         */
/*                                                                        */
/*        G(k,p,q) = \int_{-1}^1 t^k/|tp+q|^2 dt                          */
/*                                                                        */
/*        by use of its antiderivative.                                   */
/**************************************************************************/
double dlp(int k, double* p, double* q);


/******************************* computeKij *******************************/
/* INPUT: double* I0,I1 = pointers to the 2 summands I0,I1 which are      */
/*        needed for assembling one K_ij-Entry                            */
/*        double eta = admissibility constant greater or equal to 0       */
/*        double a0,a1,b0,b1,c0,c1,d0,d1 = coordinates of the points      */
/*        A,B,C,D                                                         */
/* OUTPUT: None, computeKij calls "computeKijAnalytic",                   */
/*        "computeKijSwappedAnalytic","computeKijSemianalytic" or         */
/*        "computeKijSwappedSemianalytic", output is stored in I0 and I1. */
/* DESCRIPTION: This function computes one entry of the matrix K. It      */
/*        decides whether semi analytic or analytic computation should be */
/*        chosen via the criterion:                                       */
/*                                                                        */
/*        \dist(T_i, T_j)* eta >= \min\{\diam(T_i), \diam(T_j)            */
/*                                                                        */
/*        FALSE resp. eta==0: compute analytically TRUE: compute semi     */
/*        analytically                                                    */
/*                                                                        */
/*        It furthermore ensures that the outer interval Ti is smaller    */
/*        than the inner one Tj. If this is not the case                  */
/*        "computeKijSwapped(Semi-)Analytic" is called.                   */
/*                                                                        */
/*        I0 = const*int_{-1}^1\int_{-1}^1 f(s,t) dtds I1 =               */
/*        const*int_{-1}^1\int_{-1}^1 f(s,t)*t dtds with: const =         */
/*        -1/(2pi)*|Ti|/2*|Tj|/2 and f(s,t)=<su+tv+w,n>/|su+tv+w|^2       */
/*                                                                        */
/*        All types of computation use "dlp" to compute I0,I1, which are  */
/*        computed in "computeKij(Semi-)Analytic" or                      */
/*        "computeKijSemianalytic" and assembled to one Kij-Entry in      */
/*        "computeK".                                                     */
/**************************************************************************/
void computeKij(double* I0, double* I1, double eta,
                double a0, double a1, double b0, double b1, 
                double c0, double c1, double d0, double d1);


/*************************** computeKijAnalytic ***************************/
/* INPUT: double* I0,I1 = pointers to the 2 summands I0,I1 which are      */
/*        needed for assembling one K_ij-Entry                            */
/*        double eta = admissibility constant greater or equal to 0       */
/*        double a0,a1,b0,b1,c0,c1,d0,d1 = coordinates of the points      */
/*        A,B,C,D                                                         */
/* OUTPUT: None, the output is stored in I0 and I1                        */
/* DESCRIPTION: Analytic computation of K_{ij} by use of antiderivatives. */
/*        I0 and I1 are solved with "dlp", whereas u=A-B, v=D-C and       */
/*        w=C+D-(A+B).                                                    */
/**************************************************************************/
void computeKijAnalytic(double* I0, double* I1,
                double a0, double a1, double b0, double b1, 
                double c0, double c1, double d0, double d1);


/*********************** computeKijSwappedAnalytic ************************/
/* INPUT: double* I0,I1 = pointers to the 2 summands I0,I1 which are      */
/*        needed for assembling one K_ij-Entry                            */
/*        double eta = admissibility constant greater or equal to 0       */
/*        double a0,a1,b0,b1,c0,c1,d0,d1 = coordinates of the points      */
/*        A,B,C,D                                                         */
/* OUTPUT: None, the output is stored in I0 and I1                        */
/* DESCRIPTION: Computes the entry K_{ij} similar to computeKijAnalytic,  */
/*        but the integration order is swapped, i.e. u=D-C, v=A-B and     */
/*        w=C+D-(A+B).                                                    */
/**************************************************************************/
void computeKijSwappedAnalytic(double* I0, double* I1,
                double a0, double a1, double b0, double b1, 
                double c0, double c1, double d0, double d1);


/************************* computeKijSemianalytic *************************/
/* INPUT: double* I0,I1 = pointers to the 2 summands I0,I1 which are      */
/*        needed for assembling one K_ij-Entry                            */
/*        double eta = admissibility constant greater or equal to 0       */
/*        double a0,a1,b0,b1,c0,c1,d0,d1 = coordinates of the points      */
/*        A,B,C,D                                                         */
/* OUTPUT: None, the output is stored in I0 and I1.                       */
/* DESCRIPTION: The entries of the K-matrix are calculated semi analytic. */
/*                                                                        */
/*        The outer integration (Ti) is approximated with 16-point        */
/*        Gaussian quadrature and the inner one (Tj) is computed          */
/*        analytically, i.e. u=A-B, v=D-C and w=C+D-(A+B).                */
/**************************************************************************/
void computeKijSemianalytic(double* I0, double* I1, 
		double a0, double a1, double b0, double b1, 
                double c0, double c1, double d0, double d1);

/********************* computeKijSwappedSemianalytic **********************/
/* INPUT: double* I0,I1 = pointers to the 2 summands I0,I1 which are      */
/*        needed for assembling one K_ij-Entry                            */
/*        double eta = admissibility constant greater or equal to 0       */
/*        double a0,a1,b0,b1,c0,c1,d0,d1 = coordinates of the points      */
/*        A,B,C,D                                                         */
/* OUTPUT: None, the output is stored in I0 and I1.                       */
/* DESCRIPTION: Computes the K_ij-Entry similar to                        */
/*        computeKijSemianalytic, but the integration order is swapped,   */
/*        i.e. u=D-C, v=A-B and w=C+D-(A+B).                              */
/**************************************************************************/
void computeKijSwappedSemianalytic(double* I0, double* I1, 
		double a0, double a1, double b0, double b1, 
                double c0, double c1, double d0, double d1);

#endif

