/**************************************************************************/
/* DESCRIPTION: This file is part of the HILBERT program package for the  */
/*        numerical solution of the Laplace equation with mixed boundary  */
/*        conditions by use of BEM in 2D. It provides functions to        */
/*        compute the Galerkin-Matrix K given by                          */
/*                                                                        */
/*        K_ij = -1/(2pi)\int_{T_i}\int_{supp phi_j} <y-x,n>/|y-x|^2      */
/*        phi_j(y)ds_y ds_x.                                              */
/*                                                                        */
/*        This file contains only the implementation. For detailed        */
/*        documentation see doubleLayerPotential.h                        */
/**************************************************************************/
/* VERSION: 3.1                                                           */
/**************************************************************************/
/* (C) 2009-2013 HILBERT-Team '09, '10, '12                               */
/* support + bug report:  hilbert@asc.tuwien.ac.at                        */
/**************************************************************************/
#include <math.h>
#include <assert.h>

#include "constants.h"
#include "doubleLayerPotential.h"
#include "gaussQuadrature.h"
#include "geometry.h"

double dlp(int k, double* p, double* q)
{
  double a = p[0]*p[0]+p[1]*p[1];                /* a = |p|^2   */
  double b = 2*( p[0]*q[0]+p[1]*q[1] );          /* b = 2*<p,q> */
  double c = q[0]*q[0]+q[1]*q[1];                /* c = |q|^2   */
  double D = 4*a*c-b*b;
  double root_D = 0;
  double c_minus_a = c-a;
  double G0 = 0;
  double G1 = 0;

  assert(D>=-EPS*4*a*c);
  if (D > EPS*4*a*c)
    root_D = sqrt(D);
  else
    D = 0.0;

  if (D == 0.0)
  {
    G0 = 2./c_minus_a;
  }   
  else
  {
    if (fabs(c_minus_a) < EPS*fabs(c))
      G0 = M_PI/root_D;
    else if (a < c)
      G0 = 2.*atan(root_D/c_minus_a)/root_D;
    else
      G0 = 2.*(atan(root_D/c_minus_a)+M_PI)/root_D;
  }

  if (k >= 1)
  {
    G1 = -b*G0;
    if (a+b+c > EPS*a)
      G1 += log(a+b+c);
      
    if (a-b+c > EPS*a)
      G1 -= log(a-b+c);

    G1 /= (2.*a);
    
    if (k == 2)
      return (2.-b*G1-c*G0)/a;
    
    return G1;
  }

  return G0;
}


void computeKij(double* I0, double* I1, double eta,
                double a0, double a1, double b0, double b1, 
                double c0, double c1, double d0, double d1)
{
  int swap = 0;
  double tmp = 0.0;
  double hi = (b0-a0)*(b0-a0)+(b1-a1)*(b1-a1);
  double hj = (d0-c0)*(d0-c0)+(d1-c1)*(d1-c1);
  
  if ((hi-hj)/hi > EPS)
  {
    swap = 1;     /* swap Tj and Ti */
    tmp = hi;
    hi = hj;
    hj = tmp;
  }

  if (eta == 0)   /* fully analytic computation */
  {
    if (swap == 0) 
      computeKijAnalytic(I0,I1,a0,a1,b0,b1,c0,c1,d0,d1);
    else
      computeKijSwappedAnalytic(I0,I1,a0,a1,b0,b1,c0,c1,d0,d1);
  }
  else            /* fully analytic or semianalytic */
  {
    if(distanceSegmentToSegment(a0,a1,b0,b1,c0,c1,d0,d1)*eta >= sqrt(hi))
    {
      if (swap == 0)
        computeKijSemianalytic(I0,I1,a0,a1,b0,b1,c0,c1,d0,d1);
      else
        computeKijSwappedSemianalytic(I0,I1,a0,a1,b0,b1,c0,c1,d0,d1);
    }
    else
    {
      if (swap == 0)
        computeKijAnalytic(I0,I1,a0,a1,b0,b1,c0,c1,d0,d1);
      else
        computeKijSwappedAnalytic(I0,I1,a0,a1,b0,b1,c0,c1,d0,d1);
    }
  }
}


void computeKijAnalytic(double* I0, double* I1, 
               double a0, double a1, double b0, double b1, 
               double c0, double c1, double d0, double d1)
{
  double lambda=0.0, mu=0.0;
  double det=0.0;
  
  double n[2]={0.,0.}, u[2]={0.,0.}, v[2]={0.,0.}, w[2]={0.,0.};
  double w_plus_u[2]={0.,0.}, w_minus_u[2]={0.,0.};
  double w_plus_v[2]={0.,0.}, w_minus_v[2]={0.,0.};

  double dot_u_n=0.0, dot_w_n=0.0; 
  double dot_w_plus_u_n=0.0, dot_w_minus_u_n=0.0;

  /* hi=norm2(b-a)^2 ; hj=norm2(d-c)^2 */
  double hi = (b0-a0)*(b0-a0)+(b1-a1)*(b1-a1); 
  double hj = (d0-c0)*(d0-c0)+(d1-c1)*(d1-c1); 
  
  n[0] = (d1-c1)/sqrt(hj);   n[1] = (c0-d0)/sqrt(hj);  /* normal vector */
  u[0] = a0-b0;              u[1] = a1-b1;             /* u=a-b         */
  v[0] = d0-c0;              v[1] = d1-c1;             /* v=d-c         */

  w[0] = c0+d0-a0-b0;        w[1] = c1+d1-a1-b1;       /* w=c+d-a-b     */
  w_plus_u[0] = w[0]+u[0];   w_plus_u[1] = w[1]+u[1];  /* w_plus_u=w+u  */
  w_minus_u[0] = w[0]-u[0];  w_minus_u[1] = w[1]-u[1]; /* w_minus_u=w-u */

  w_plus_v[0] = w[0]+v[0];   w_plus_v[1] = w[1]+v[1];
  w_minus_v[0] = w[0]-v[0];  w_minus_v[1] = w[1]-v[1];

  dot_u_n = u[0]*n[0]+u[1]*n[1];                       /* dot_u_n=<u,n> */
  dot_w_n = w[0]*n[0]+w[1]*n[1];             /* dot_w_plus_u_n=<w+u,n>  */
  dot_w_plus_u_n = w_plus_u[0]*n[0]+w_plus_u[1]*n[1];     
  dot_w_minus_u_n = w_minus_u[0]*n[0]+w_minus_u[1]*n[1];

  det = u[0]*v[1]-u[1]*v[0];
  
  if (fabs(det) <= EPS*sqrt(hi*hj))           /* u,v linearly dependent */
  {
    if (fabs(u[0]) > fabs(u[1]))
      mu = v[0]/u[0]; 
    else
      mu = v[1]/u[1];

    *I0 = dot_w_n*( dlp(0,u,w_plus_v)+dlp(0,u,w_minus_v)
          + mu*(dlp(1,v,w_minus_u)-dlp(1,v,w_plus_u)) );
    *I1 = dot_w_n*( dlp(0,u,w_plus_v)-dlp(0,u,w_minus_v)
          + mu*(dlp(2,v,w_minus_u)-dlp(2,v,w_plus_u)) )*0.5;
  }
  else                                      /* u,v linearly independent */
  {
    if (a0 == d0 && a1 == d1)
    {
      *I0 = 2*( dot_w_plus_u_n*dlp(0,v,w_plus_u)+dot_u_n*dlp(1,u,w_minus_v)
                + dot_w_n*dlp(0,u,w_minus_v) );
      *I1 =     dot_w_plus_u_n*dlp(1,v,w_plus_u)-dot_u_n*dlp(1,u,w_minus_v)
                - dot_w_n*dlp(0,u,w_minus_v) + 0.5*(*I0);
    }
    else if (b0 == c0 && b1 == c1)
    {
      *I0 = 2*( dot_w_minus_u_n*dlp(0,v,w_minus_u)
                + dot_u_n*dlp(1,u,w_plus_v)+dot_w_n*dlp(0,u,w_plus_v) );
      *I1 =     dot_w_minus_u_n*dlp(1,v,w_minus_u)
                + dot_u_n*dlp(1,u,w_plus_v)+dot_w_n*dlp(0,u,w_plus_v) 
                - 0.5*(*I0);
    }
    else
    {
      mu     = (w[0]*v[1]-w[1]*v[0])/det;
      lambda = (u[0]*w[1]-u[1]*w[0])/det;
     
      *I0 = (mu+1)*dot_w_plus_u_n*dlp(0,v,w_plus_u)
            - (mu-1)*dot_w_minus_u_n*dlp(0,v,w_minus_u)
            + (lambda+1)*(dot_u_n*dlp(1,u,w_plus_v)
   	                  + dot_w_n*dlp(0,u,w_plus_v))
            - (lambda-1)*(dot_u_n*dlp(1,u,w_minus_v) 
                          + dot_w_n*dlp(0,u,w_minus_v));
      *I1 = 0.5*( (mu+1)*dot_w_plus_u_n*dlp(1,v,w_plus_u)
                 - (mu-1)*dot_w_minus_u_n*dlp(1,v,w_minus_u)
                 + (lambda+1)*(dot_u_n*dlp(1,u,w_plus_v)
   	                       + dot_w_n*dlp(0,u,w_plus_v))
                 + (lambda-1)*(dot_u_n*dlp(1,u,w_minus_v)
                               + dot_w_n*dlp(0,u,w_minus_v)) 
                 - lambda*(*I0) ); 
    }
  }
  *I0 *= -0.125*sqrt(hi*hj)/M_PI;
  *I1 *= -0.125*sqrt(hi*hj)/M_PI;
}


void computeKijSwappedAnalytic(double* I0, double* I1, 
               double a0, double a1, double b0, double b1, 
               double c0, double c1, double d0, double d1)
{
  double lambda = 0;
  double mu = 0;
  double det = 0;

  /*hi=norm2(b-a)^2 ; hj=norm2(d-c)^2 */
  double hi = (b0-a0)*(b0-a0)+(b1-a1)*(b1-a1); 
  double hj = (d0-c0)*(d0-c0)+(d1-c1)*(d1-c1); 
  
  double n[2], u[2], v[2], w[2];
  double w_plus_u[2], w_minus_u[2], w_plus_v[2], w_minus_v[2];
  double dot_v_n, dot_w_n, dot_w_plus_v_n, dot_w_minus_v_n;

  n[0] = (d1-c1)/sqrt(hj);    n[1] = (c0-d0)/sqrt(hj);  /* normal vector */
  u[0] = d0-c0;               u[1] = d1-c1;             /* u=d-c         */
  v[0] = a0-b0;               v[1] = a1-b1;             /* v=a-b         */

  w[0] = c0+d0-a0-b0;         w[1] = c1+d1-a1-b1;       /* w=c+d-a-b     */
  w_plus_u[0] = w[0]+u[0];    w_plus_u[1] = w[1]+u[1];  /* w_plus_u=w+u  */
  w_minus_u[0] = w[0]-u[0];   w_minus_u[1] = w[1]-u[1]; /* w_minus_u=w-u */

  w_plus_v[0] = w[0]+v[0];    w_plus_v[1] = w[1]+v[1];
  w_minus_v[0] = w[0]-v[0];   w_minus_v[1] = w[1]-v[1];

  dot_v_n = v[0]*n[0]+v[1]*n[1];                        /* dot_u_n=<u,n> */
  dot_w_n = w[0]*n[0]+w[1]*n[1];               /* dot_w_plus_u_n=<w+u,n> */
  dot_w_plus_v_n = w_plus_v[0]*n[0]+w_plus_v[1]*n[1];     
  dot_w_minus_v_n = w_minus_v[0]*n[0]+w_minus_v[1]*n[1];

  det = u[0]*v[1]-u[1]*v[0];
  
  if(fabs(det)<=EPS*sqrt(hi*hj))               /* u,v linearly dependent */
  {
    if(fabs(u[0])>fabs(u[1]))
      mu = v[0]/u[0];
    else
      mu = v[1]/u[1];

    *I0 = dot_w_n*( dlp(0,u,w_plus_v)+dlp(0,u,w_minus_v)
          + mu*(dlp(1,v,w_minus_u)-dlp(1,v,w_plus_u)) );
    *I1 = dot_w_n*( dlp(1,u,w_plus_v)+dlp(1,u,w_minus_v)
		    + mu*( -dlp(1,v,w_minus_u)-dlp(1,v,w_plus_u)        
                           +0.5*( dlp(0,u,w_plus_v)-dlp(0,u,w_minus_v)
	                        + mu*(dlp(2,v,w_minus_u)-dlp(2,v,w_plus_u)))));
  }
  else                                       /* u,v linearly independent */
  {
    if (a0 == d0 && a1 == d1)
    {
      *I0 = 2*( dot_v_n*dlp(1,v,w_minus_u)+dot_w_n*dlp(0,v,w_minus_u)
                + dot_w_plus_v_n*dlp(0,u,w_plus_v) );
      *I1 =     dot_w_plus_v_n*dlp(1,u,w_plus_v)-dot_v_n*dlp(1,v,w_minus_u)
                - dot_w_n*dlp(0,v,w_minus_u) + 0.5*(*I0);
    }
    else if (b0 == c0 && b1 == c1)
    {
      *I0 = 2*( dot_v_n*dlp(1,v,w_plus_u)
                +dot_w_n*dlp(0,v,w_plus_u)
                +dot_w_minus_v_n*dlp(0,u,w_minus_v) );
      *I1 =     dot_v_n*dlp(1,v,w_plus_u)+dot_w_n*dlp(0,v,w_plus_u)
                + dot_w_minus_v_n*dlp(1,u,w_minus_v) - 0.5*(*I0);
    }
    else
    {
      mu     = (w[0]*v[1]-w[1]*v[0])/det;
      lambda = (u[0]*w[1]-u[1]*w[0])/det;
     
      *I0 = (mu+1)*(dot_v_n*dlp(1,v,w_plus_u)+dot_w_n*dlp(0,v,w_plus_u))
	    - (mu-1)*(dot_v_n*dlp(1,v,w_minus_u)+dot_w_n*dlp(0,v,w_minus_u))
            + (lambda+1)*dot_w_plus_v_n*dlp(0,u,w_plus_v)
	    - (lambda-1)*dot_w_minus_v_n*dlp(0,u,w_minus_v);
      *I1 = 0.5*( (mu+1)*(dot_v_n*dlp(1,v,w_plus_u)
                            +dot_w_n*dlp(0,v,w_plus_u))
	         +(mu-1)*(dot_v_n*dlp(1,v,w_minus_u)+dot_w_n*dlp(0,v,w_minus_u))
                 +(lambda+1)*dot_w_plus_v_n*dlp(1,u,w_plus_v)
	         -(lambda-1)*dot_w_minus_v_n*dlp(1,u,w_minus_v)-mu*(*I0));
    }
  }
  *I0 *= -0.125*sqrt(hi*hj)/M_PI;
  *I1 *= -0.125*sqrt(hi*hj)/M_PI;
}


void computeKijSemianalytic(double* I0, double* I1, 
               double a0, double a1, double b0, double b1, 
               double c0, double c1, double d0, double d1)
{
  /*hi=norm2(b-a)^2 hj=norm2(d-c)^2 */
  double hi = (b0-a0)*(b0-a0)+(b1-a1)*(b1-a1); 
  double hj = (d0-c0)*(d0-c0)+(d1-c1)*(d1-c1); 
  
  int i=0;
  double n[2], u[2], v[2], w[2], z[2];
  double I0tmp = 0.0;
  double I1tmp = 0.0;
  double dot_z_n = 0.0;

  /* 16-point Gaussian quadrature on [-1,1] */
  const int order = 16;
  const double* gauss_weight = getGaussWeights(order);
  const double* gauss_point  = getGaussPoints(order);

  n[0] = (d1-c1)/sqrt(hj); n[1] = (c0-d0)/sqrt(hj);     /* normal vector */
  u[0] = a0-b0;            u[1] = a1-b1;                /* u=a-b         */
  v[0] = d0-c0;            v[1] = d1-c1;                /* v=d-c         */
  w[0] = c0+d0-a0-b0;      w[1] = c1+d1-a1-b1;          /* w=c+d-a-b     */

  for (i=0;i<order;++i)
  {
    z[0] = gauss_point[i]*u[0]+w[0];
    z[1] = gauss_point[i]*u[1]+w[1];
   
    dot_z_n = z[0]*n[0]+z[1]*n[1];
   
    I0tmp += gauss_weight[i]*dot_z_n*dlp(0,v,z);
    I1tmp += gauss_weight[i]*dot_z_n*dlp(1,v,z);
  }

  *I0 = -0.125*sqrt(hi*hj)*I0tmp/M_PI;
  *I1 = -0.125*sqrt(hi*hj)*I1tmp/M_PI;
}

void computeKijSwappedSemianalytic(double* I0, double* I1, 
               double a0, double a1, double b0, double b1, 
               double c0, double c1, double d0, double d1)
{
  /*hi=norm2(b-a)^2 ; hj=norm2(d-c)^2 */
  double hi = (b0-a0)*(b0-a0)+(b1-a1)*(b1-a1); 
  double hj = (d0-c0)*(d0-c0)+(d1-c1)*(d1-c1); 
  
  int i=0;
  double n[2], u[2], v[2], w[2], z[2];
  double dot_v_n, dot_w_n;
  double I0tmp = 0.0;
  double I1tmp = 0.0;

  /* 16-point Gaussian quadrature on [-1,1] */
  const int order = 16;
  const double* gauss_weight = getGaussWeights(order);
  const double* gauss_point  = getGaussPoints(order);


  n[0] = (d1-c1)/sqrt(hj); n[1] = (c0-d0)/sqrt(hj);     /* normal vector */
  u[0] = d0-c0;            u[1] = d1-c1;                /* u=d-c         */
  v[0] = a0-b0;            v[1] = a1-b1;                /* v=a-b         */
  w[0] = c0+d0-a0-b0;      w[1] = c1+d1-a1-b1;          /* w=c+d-a-b     */

  dot_v_n = v[0]*n[0]+v[1]*n[1];
  dot_w_n = w[0]*n[0]+w[1]*n[1];
 
  for (i=0;i<order;++i)
  {
    z[0] = gauss_point[i]*u[0]+w[0];
    z[1] = gauss_point[i]*u[1]+w[1];
   
    I0tmp += gauss_weight[i]*( dot_v_n*dlp(1,v,z) + dot_w_n*dlp(0,v,z) );
    I1tmp += gauss_weight[i]*gauss_point[i]*( dot_v_n*dlp(1,v,z) 
                                             + dot_w_n*dlp(0,v,z) );
  }

  *I0 = -0.125*sqrt(hi*hj)*I0tmp/M_PI;
  *I1 = -0.125*sqrt(hi*hj)*I1tmp/M_PI;
}

