/**************************************************************************/
/* DESCRIPTION: This file is part of the HILBERT program package for the  */
/*        numerical solution of the Laplace equation with mixed boundary  */
/*        conditions by use of BEM in 2D. It provides functions for the   */
/*        calculation of the hypersingular integral operator matrix W     */
/*        that is defined by                                              */
/*                                                                        */
/*        W_{jk} = < W phi_j, phi_k >                                     */
/*                                                                        */
/*        with the hat functions phi_i corresponding to a piece-wise      */
/*        affine function that is linear on any element of the boundary   */
/*        mesh and that is 1 on the i-th node z_i and 0 on any other      */
/*        node. The matrix W is in fact calculated using the following    */
/*        relation between W and the simple-layer potential V:            */
/*                                                                        */
/*        <Wu, v> = <Vu', v'>                                             */
/*                                                                        */
/*        In this formula, u' and v' denote the arc-length derivative of  */
/*        u and v respectively.                                           */
/**************************************************************************/
/* VERSION: 3.1                                                           */
/**************************************************************************/
/* (C) 2009-2013 HILBERT-Team '09, '10, '12                               */
/* support + bug report:  hilbert@asc.tuwien.ac.at                        */
/**************************************************************************/
#ifndef _BUILDW_H_GUARD_
#define _BUILDW_H_GUARD_

#include "mex.h"

/****************************** mexFunction *******************************/
/* USAGE: W = buildW(coordinates,elements [,eta])                         */
/* INPUT: coordinates and elements are matrices describing the            */
/*        triangulation. The optional parameter eta>=0 is used to         */
/*        determine the set of admissible elements. If two elements are   */
/*        admissible, semi analytic computation is performed. Otherwise   */
/*        the corresponding entry of V is computed fully analytically.    */
/*        Pass eta=0 to enforce fully analytic computation of V.          */
/* OUTPUT: The Galerkin-Matrix W_{jk} = < Wpsi_k , psi_j >                */
/* DESCRIPTION: This function is a MEX-wrapper around computeW. For a     */
/*        more detailed documentation see buildW.m                        */
/**************************************************************************/
void mexFunction(int nlhs, mxArray* plhs[],
                  int nrhs, const mxArray* prhs[]);

/******************************** computeW ********************************/
/* INPUT: W is a nE x nE matrix. nC and nE are the number of coordinates  */
/*        and elements, respectively. coordinates and elements contain    */
/*        the coordinates and elements of the boundary mesh. eta is the   */
/*        admissibility constant. It is greater or equal than 0.          */
/* OUTPUT: None, This function modifies the matrix W.                     */
/* DESCRIPTION: This function assembles the matrix W by calling           */
/*        computeWij for each pair of boundary elements (Ti, Tj) and      */
/*        storing the result in W. It uses the symmetry of W to reduce    */
/*        the build time.                                                 */
/**************************************************************************/
void computeW(double* W, int nC, int nE,
              const double* coordinates, const double* elements,
              double eta);

#endif

