/**************************************************************************/
/* DESCRIPTION: This file is part of the HILBERT program package for the  */
/*        numerical solution of the Laplace equation with mixed boundary  */
/*        conditions by use of BEM in 2D. It provides functions for the   */
/*        calculation of the hypersingular integral operator matrix W     */
/*        that is defined by                                              */
/*                                                                        */
/*        W_{jk} = < W phi_j, phi_k >                                     */
/*                                                                        */
/*        with the hat functions phi_i corresponding to a piece-wise      */
/*        affine function that is linear on any element of the boundary   */
/*        mesh and that is 1 on the i-th node z_i and 0 on any other      */
/*        node. The matrix W is in fact calculated using the following    */
/*        relation between W and the simple-layer potential V:            */
/*                                                                        */
/*        <Wu, v> = <Vu', v'>                                             */
/*                                                                        */
/*        In this formula, u' and v' denote the arc-length derivative of  */
/*        u and v respectively.                                           */
/**************************************************************************/
/* VERSION: 3.1                                                           */
/**************************************************************************/
/* (C) 2009-2013 HILBERT-Team '09, '10, '12                               */
/* support + bug report:  hilbert@asc.tuwien.ac.at                        */
/**************************************************************************/
#include "mex.h"
#include <math.h>

#include "buildW.h"
#include "constants.h"
#include "singleLayerPotential.h"

#ifdef HILTHREADS
#include "threadedW.h"
#endif

void mexFunction(int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[])
{
  const char* function_name = mexFunctionName();
  char error_message[255];
  int nC, nE;
  double* W = NULL;
  const double* elements;
  const double* coordinates;
  double eta = DEFAULT_ETA;

  if (nlhs != 1)
  {
    sprintf(error_message,
      "%s expects exactly one left-hand side argument.\n"
      "Use either W = %s(coordinates, elements) or \n"
      "W = %s(coordinates, elements, eta).",
      function_name, function_name, function_name);
    mexErrMsgTxt(error_message);
  }

  /* Read input data */
  switch (nrhs)
  {
    case 3:
      eta = extract_scalar_or(prhs[2], DEFAULT_ETA);
    case 2:
      coordinates = mxGetPr(prhs[0]);
      nC = mxGetM(prhs[0]);
      elements = mxGetPr(prhs[1]);
      nE = mxGetM(prhs[1]);
      break;
    default:
      sprintf(error_message,
        "%s expects either two or three right-hand side arguments.\n"
        "Use either W = %s(coordinates, elements) or \n"
        "W = %s(coordinates, elements, eta).",
        function_name, function_name, function_name);
      mexErrMsgTxt(error_message);
      return;
  }

  /* Allocate memory for output data */
  plhs[0] = mxCreateDoubleMatrix(nC, nC, mxREAL);
  W = mxGetPr(plhs[0]);

#ifdef HILTHREADS
  computeWThreaded(W, coordinates, elements, nC, nE, eta);
#else
  computeW(W, nC, nE, coordinates, elements, eta);
#endif
}

void computeW(double* W, int nC, int nE,
              const double* coordinates, const double* elements,
              double eta)
{
  int i = 0, j = 0;
  int aidx = 0, bidx = 0, cidx = 0, didx = 0;
  double a0=0., a1=0., b0=0., b1=0., c0=0., c1=0., d0=0., d1=0.;
  double tmp = 0.;

  for (i = 0; i < nE; ++i)
  {
    aidx = (int) elements[i] - 1;
    a0 = coordinates[aidx];
    a1 = coordinates[aidx+nC];

    bidx = (int) elements[i+nE] - 1;
    b0 = coordinates[bidx];
    b1 = coordinates[bidx+nC];

    for (j = 0; j < nE; ++j)
    {
      cidx = (int) elements[j] - 1;
      c0 = coordinates[cidx];
      c1 = coordinates[cidx+nC];

      didx = (int) elements[j+nE] - 1;
      d0 = coordinates[didx];
      d1 = coordinates[didx+nC];

      tmp = computeWij(a0, a1, b0, b1, c0, c1, d0, d1, eta);

      W[aidx + cidx*nC] += tmp;
      W[aidx + didx*nC] -= tmp;
      W[bidx + cidx*nC] -= tmp;
      W[bidx + didx*nC] += tmp;
    }
  }
}

