/**************************************************************************/
/* DESCRIPTION: This file is part of the HILBERT program package for the  */
/*        numerical solution of the Laplace equation with mixed boundary  */
/*        conditions by use of BEM in 2D. It provides functions for the   */
/*        computation of the simple-layer potential matrix V defined by   */
/*                                                                        */
/*                                                                        */
/*        V_{jk} = < Vchi_j , chi_k >                                     */
/*                                                                        */
/*        with characteristic functions chi_j corresponding to a line     */
/*        segment T_j. The simple-layer potential is defined by           */
/*                                                                        */
/*        Vchi_j = -1/(2pi) \int_{T_j} log|x-y| ds_y.                     */
/*                                                                        */
/**************************************************************************/
/* VERSION: 3.1                                                           */
/**************************************************************************/
/* (C) 2009-2013 HILBERT-Team '09, '10, '12                               */
/* support + bug report:  hilbert@asc.tuwien.ac.at                        */
/**************************************************************************/
#ifndef _BUILDV_H_GUARD_
#define _BUILDV_H_GUARD_

#include "mex.h"

/****************************** mexFunction *******************************/
/* USAGE: V = buildV(coordinates,elements [,eta])                         */
/* INPUT: coordinates and elements are matrices describing the            */
/*        triangulation. The optional parameter eta>=0 is used to         */
/*        determine the set of admissible elements. If two elements are   */
/*        admissible, semi analytic computation is performed. Otherwise   */
/*        the corresponding entry of V is computed fully analytically.    */
/*        Pass eta=0 to enforce fully analytic computation of V.          */
/* OUTPUT: The Galerkin-Matrix V_{jk} = < Vchi_j , chi_k >                */
/* DESCRIPTION: This function is a MEX-wrapper around computeV. For a     */
/*        more detailed documentation see buildV.m                        */
/**************************************************************************/
void mexFunction(int nlhs, mxArray* plhs[],
                  int nrhs, const mxArray* prhs[]);

/******************************** computeV ********************************/
/* INPUT: V is a ne x ne matrix. nc and ne are the number of coordinates  */
/*        and elements, respectively. coordinates and elements contain    */
/*        the coordinates and elements of the boundary mesh. eta is the   */
/*        admissibility constant. It is greater or equal than 0.          */
/* OUTPUT: None, This function modifies the matrix V.                     */
/* DESCRIPTION: This function uses computeVHelper to assemble the single- */
/*        layer potential matrix V. Depending on whether threading is     */
/*        enabled or not, this function either spawns a number of         */
/*        computeVWorkers that call computeVHelper or it calls the        */
/*        function directly.                                              */
/**************************************************************************/
void computeV(double* V, int nc, int ne,
              const double* coordinates, const double* elements,
              double eta);

#endif

