/**************************************************************************/
/* DESCRIPTION: This file is part of the HILBERT program package for the  */
/*        numerical solution of the Laplace equation with mixed boundary  */
/*        conditions by use of BEM in 2D. It provides functions for the   */
/*        computation of the single-layer potential matrix V defined by   */
/*                                                                        */
/*                                                                        */
/*        V_{jk} = < Vchi_j , chi_k >                                     */
/*                                                                        */
/*        with characteristic functions chi_j corresponding to a line     */
/*        segment T_j. The single-layer potential is defined by           */
/*                                                                        */
/*        Vchi_j = -1/(2pi) \int_{T_j} log|x-y| ds_y.                     */
/*                                                                        */
/*        This file contains only the implementation. For extensive       */
/*        documentation consult the corresponding header-file.            */
/**************************************************************************/
/* VERSION: 3.1                                                           */
/**************************************************************************/
/* (C) 2009-2013 HILBERT-Team '09, '10, '12                               */
/* support + bug report:  hilbert@asc.tuwien.ac.at                        */
/**************************************************************************/
#include "mex.h"
#include <math.h>

#include "buildV.h"
#include "constants.h"
#include "singleLayerPotential.h"

#ifdef HILTHREADS
#  include "threadedV.h"
#endif

void mexFunction(int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[])
{
  const char* function_name = mexFunctionName();
  char error_message[255];
  int nc = 0, ne = 0;
  double* V = NULL;
  const double* elements;
  const double* coordinates;
  double eta = DEFAULT_ETA;

  if (nlhs != 1)
  {
    sprintf(error_message,
            "Invalid number of output arguments. Use either\n"
            "  V = %s(coordinates,elements)\n"
            "or\n"
            "  V = %s(coordinates,elements,eta)\n",
            function_name,function_name);
    mexErrMsgTxt(error_message);
  }

  /* Read input data */
  switch (nrhs)
  {
    case 3:
      eta = extract_scalar_or(prhs[2], DEFAULT_ETA);
    case 2:
      coordinates = (const double*) mxGetPr(prhs[0]);
      nc          = mxGetM(prhs[0]);
      elements    = (const double*) mxGetPr(prhs[1]);
      ne          = mxGetM(prhs[1]);
      break;
    default:
      sprintf(error_message,
              "Invalid number of input arguments (%d). Use either\n"
              "  V = %s(coordinates, elements)\n"
              "or\n"
              "  V = %s(coordinates, elements, eta)\n",
              nrhs, function_name, function_name);
      mexErrMsgTxt(error_message);
      return;
  }

  if (eta < 0)
  {
    sprintf(error_message,
      "In %s, ETA is less than zero (ETA=%f).\n"
      "Please choose a value that is greater than or equal to 0.\n",
      function_name, eta);
    mexErrMsgTxt(error_message);
  }

  /* Allocate output data */
  plhs[0] = mxCreateDoubleMatrix(ne,ne,mxREAL);
  V = mxGetPr(plhs[0]);

#ifdef HILTHREADS
  if (ne < HILTHR_V_MINROW_PER_WORKER)
    computeV(V, nc, ne, coordinates, elements, eta);
  else
    computeVThreaded(V, nc, ne, coordinates, elements, eta);
#else
  computeV(V, nc, ne, coordinates, elements, eta);
#endif
}

void computeV(double* V, int nc, int ne,
               const double* coordinates, const double* elements,
               double eta)
{
  int    i = 0, j = 0;
  int    aidx = 0, bidx = 0, cidx = 0, didx = 0;
  double a0 = 0., a1 = 0., b0 = 0., b1 = 0.;
  double c0 = 0., c1 = 0., d0 = 0., d1 = 0.;

  assert(eta >= 0);

  for (i=0; i<ne; ++i)
  {
    aidx = (int) elements[i]-1;      /* 1st node of element T_i = [A,B] */
    a0 = coordinates[aidx];
    a1 = coordinates[aidx+nc];

    bidx = (int) elements[i+ne]-1;   /* 2nd node of element T_i = [A,B] */
    b0 = coordinates[bidx];
    b1 = coordinates[bidx+nc];

    for (j=i; j<ne; ++j)
    {
      cidx = (int) elements[j]-1;    /* 1st node of element T_j = [C,D] */
      c0 = coordinates[cidx];
      c1 = coordinates[cidx+nc];

      didx = (int) elements[j+ne]-1; /* 2nd node of element T_j = [C,D] */
      d0 = coordinates[didx];
      d1 = coordinates[didx+nc];

      V[i + j*ne] = computeVij(a0,a1,b0,b1, c0,c1,d0,d1,eta);
      V[j + i*ne] = V[i + j*ne];
    }
  }
}

