/**************************************************************************/
/* DESCRIPTION: This file is part of the HILBERT program package for the  */
/*        numerical solution of the Laplace equation with mixed boundary  */
/*        conditions by use of BEM in 2D. It provides functions to read   */
/*        the input-parameters, which are relevant to compute the         */
/*        Galerkin-Matrix N corresponding to the Newton potential. Refer  */
/*        either to buildN.m or the file newtonPotential.h for more       */
/*        details.                                                        */
/**************************************************************************/
/* VERSION: 3.1                                                           */
/**************************************************************************/
/* (C) 2009-2013 HILBERT-Team '10, '12                                    */
/* support + bug report:  hilbert@asc.tuwien.ac.at                        */
/**************************************************************************/
#include "mex.h"
#include <math.h>

#include "buildN.h"
#include "constants.h"
#include "newtonPotential.h"

#ifdef HILTHREADS
#  include "threadedN.h"
#endif

void mexFunction(int nlhs, mxArray** plhs, int nrhs, const mxArray** prhs)
{
  const char* function_name = mexFunctionName();
  char error_message[150];
  int nC=0, nE=0, nV=0, nVol=0;
  const double* coordinates=NULL;
  const double* elements=NULL;
  const double* vertices=NULL;
  const double* triangles=NULL;
  double *N=NULL;
  double eta = DEFAULT_ETA;

  if(nlhs != 1)
  {
    sprintf(error_message,"Invalid number of output arguments. Use\n"
          "  N = %s(coordinates,elements,vertices,triangles)\n"
          "or\n"
          "  N = %s(coordinates,elements,vertices,triangles,eta)\n",
          function_name,function_name);
    mexErrMsgTxt(error_message); 
  }

  /* Read input data */
  switch (nrhs)
  {
    case 5:
      eta = extract_scalar_or(prhs[4], DEFAULT_ETA);
    case 4:
      coordinates = (const double*) mxGetPr(prhs[0]);
      nC = mxGetM(prhs[0]);
      elements = (const double*) mxGetPr(prhs[1]);
      nE = mxGetM(prhs[1]);
      vertices = (const double*) mxGetPr(prhs[2]);
      nV = mxGetM(prhs[2]);
      triangles = (const double*) mxGetPr(prhs[3]);
      nVol = mxGetM(prhs[3]);
      break;
    default:
      sprintf(error_message,
          "Invalid number of input arguments (%d). Use eiter\n"
          "  N = %s(coordinates,elements,vertices,triangles)\n"
          "or\n"
          "  N = %s(coordinates,elements,vertices,triangles,eta)\n",
          nrhs, function_name, function_name);
      mexErrMsgTxt(error_message);
      return;
  }

  if (eta < 0)
  {
    sprintf(error_message,
      "In %s, ETA is less than zero (ETA=%f).\n"
      "Please choose a value that is greater than or equal to 0.\n",
      function_name, eta);
    mexErrMsgTxt(error_message);
  }

  /* allocate output data for P0-elements */
  plhs[0] = mxCreateDoubleMatrix(nE,nVol,mxREAL);
 
  N = mxGetPr(plhs[0]);

#ifdef HILTHREADS
  computeNThreaded(N,coordinates,elements,vertices,triangles,
                    nC,nE,nV,nVol,eta);
#else
  computeN(N,coordinates,elements,vertices,triangles,nC,nE,nV,nVol,eta);
#endif
}

