/**************************************************************************/
/* DESCRIPTION: This file is part of the HILBERT program package for the  */
/*        numerical solution of the Laplace equation with mixed boundary  */
/*        conditions by use of BEM in 2D. It provides functions to read   */
/*        the input-parameters, which are relevant to compute the         */
/*        Galerkin-Matrix K corresponding to the double-layer potential.  */
/*        The matrix is given by                                          */
/*                                                                        */
/*        K_{ij} = -1/(2pi)\int_{T_i}\int_{supp phi_j} <y-x,n>/|y-x|^2    */
/*        phi_j(y)ds_y ds_x.                                              */
/*                                                                        */
/*        This file contains only the implementation. For extensive       */
/*        documentation consult the corresponding header-file.            */
/**************************************************************************/
/* VERSION: 3.1                                                           */
/**************************************************************************/
/* (C) 2009-2013 HILBERT-Team '09, '10, '12                               */
/* support + bug report:  hilbert@asc.tuwien.ac.at                        */
/**************************************************************************/
#include "mex.h"
#include <math.h>

#include "buildK.h"
#include "constants.h"
#include "doubleLayerPotential.h"

#ifdef HILTHREADS
#  include "threadedK.h"
#endif

void mexFunction(int nlhs, mxArray** plhs,
                  int nrhs, const mxArray** prhs)
{
  const char* functionName = mexFunctionName();
  char errorMessage[150];
  int nC=0, nE=0;
  double eta = DEFAULT_ETA;
  double* K=NULL;
  const double* elements=NULL;
  const double* coordinates=NULL;

  if(nlhs!=1)
  {
    sprintf(errorMessage,
            "Invalid number of output arguments. Use either\n"
            "  K = %s(coordinates,elements) or\n"
            "  K = %s(coordinates,elements,eta).\n",
            functionName,functionName);
    mexErrMsgTxt(errorMessage); 
  }

  /* read input data */
  switch(nrhs)
  {
    case 3: 
      eta = extract_scalar_or(prhs[2], DEFAULT_ETA);
    case 2:
      coordinates = (const double*) mxGetPr(prhs[0]);
      elements = (const double*) mxGetPr(prhs[1]);
      nC = mxGetM(prhs[0]);
      nE = mxGetM(prhs[1]);
      break;
    default:
      sprintf(errorMessage,
              "Invalid number of input arguments. Use either\n"
              "  K = %s(coordinates,elements) or\n"
              "  K = %s(coordinates,elements,eta).\n",
              functionName,functionName);
      mexErrMsgTxt(errorMessage); 
      break;
  }

  /* allocate output data */
  plhs[0] = mxCreateDoubleMatrix(nE,nC,mxREAL);
  K = mxGetPr(plhs[0]);

  #ifdef HILTHREADS
    computeKThreaded(K, coordinates, elements, nC, nE, eta);
  #else
    computeK(K, coordinates, elements, nC, nE, eta);
  #endif
}

void computeK(double* K, const double* coordinates, 
              const double* elements, int nC, int nE, double eta)
{
  int i=0, j=0;
  int aidx=0, bidx=0, cidx=0, didx=0;
  double a0=0.0, a1=0.0, b0=0.0, b1=0.0,
          c0=0.0, c1=0.0, d0=0.0, d1=0.0;
  double linetest1=0.0, linetest2=0.0;
  double I0=0.0, I1=0.0;
  
  for (j=0;j<nE;++j)
  {
    aidx = (int) elements[j]-1;         /* 1st node of Ti = [A,B] */
    a0 = coordinates[aidx];
    a1 = coordinates[aidx+nC];

    bidx = (int) elements[j+nE]-1;      /* 2nd node of Ti = [A,B] */
    b0 = coordinates[bidx];
    b1 = coordinates[bidx+nC];
   
    for (i=0;i<nE;++i)
    {
      cidx = (int) elements[i]-1;       /* 1st node of Tj = [C,D] */
      c0 = coordinates[cidx];
      c1 = coordinates[cidx+nC];

      didx = (int) elements[i+nE]-1;    /* 2nd node of Tj = [C,D] */
      d0 = coordinates[didx];
      d1 = coordinates[didx+nC];
      
      linetest1 = fabs( (a0-c0)*(b1-a1) - (a1-c1)*(b0-a0) );
      linetest2 = fabs( (a0-d0)*(b1-a1) - (a1-d1)*(b0-a0) );


      if( linetest1>EPS*( sqrt((a0-c0)*(a0-c0)+(a1-c1)*(a1-c1)) ) ||
          linetest2>EPS*( sqrt((a0-d0)*(a0-d0)+(a1-d1)*(a1-d1)) ) )
      {
        computeKij(&I0,&I1,eta,a0,a1,b0,b1,c0,c1,d0,d1);
        K[j+cidx*nE] += I0-I1;
        K[j+didx*nE] += I0+I1;	
      }
    }
  }
}

