function [coordinates,varargout] = refineBoundaryMesh(coordinates,varargin)
%REFINEBOUNDARYMESH   Refines a given boundary mesh.
%   REFINEBOUNDARYMESH refines a given boundary mesh via bisection of 
%   certain marked elements and guarantees the uniform K-mesh property.
%
%   Usage: [COORDINATES,ELEMENTS [,FATHER2ELEMENTS]] ...
%            = REFINEBOUNDARYMESH(COORDINATES,ELEMENTS [,MARKED_ELEMENTS])
%
%      or  [COORDINATES,DIRICHLET,NEUMANN ...
%                                  [,FATHER2DIRICHLET,FATHER2NEUMANN]]
%            = REFINEBOUNDARYMESH(COORDINATES,DIRICHLET,NEUMANN ...
%                                  [,MARKED_DIRICHLET,MARKED_NEUMANN])
%
%      or  [COORDINATES,<BOUNDARIES> [,<FATHER2BOUNDARIES>]) ...
%            = REFINEBOUNDARYMESH(COORDINATES,<BOUNDARIES> ... 
%                [,<MARKED_BOUNDARIES>])
%
%   where BOUNDARIES is a synonym for DIRICHLET, NEUMANN, ROBIN, ...
%   so that one can manage any finite number of boundary parts
%
%   Let {E1,...,EN} be a mesh described by matrices COORDINATES and 
%   ELEMENTS. The optional (m x 1)-vector MARKED_ELEMENTS contains the 
%   indices of all elements Ej which will be refined by bisection. The 
%   refined partition is returned in terms of extended matrices COORDINATES
%   and ELEMENTS. Moreover, the optional (N x 2)-matrix FATHER2ELEMENT links
%   the initial mesh with the refined mesh in the sense that 
%   FATHER2ELEMENT(j,:) contains the indices of the fine elements which are
%   the sons of the coarse-mesh element Ej. If Ej has not been refined,
%   FATHER2ELEMENT(j,1) = FATHER2ELEMENT(j,2) contains the index of Ej with
%   respect to the refined mesh.
%
%   If the optional vector MARKED_ELEMENTS is not given, the function
%   performs a uniform refinement, i.e., all elements Ej are refined.
%
%   Optionally, the boundary Gamma can be split into finitely many
%   boundaries, e.g., a Dirichlet boundary, Neumann boundary, Robin 
%   boundary, ... This partition is then given in terms of (X x 2)-matrices
%   DIRICHLET, NEUMANN, ROBIN, ... and the indices of marked elements are 
%   then given by column vectors MARKED_DIRICHLET, MARKED_NEUMANN, 
%   MARKED_ROBIN, respectively.
%
%   Throughout, the refinement procedure guarantees that the K-mesh constant 
%
%      max { diam(Ej)/diam(Ek) : Ej, Ek neighbours }
%
%   stays uniformly bounded in terms of the K-mesh constant of the initial 
%   mesh. More precisely, it ensures 
%
%      C(Mesh_ell) <= 2*C(Mesh_0)
%
%   where C(.) denotes the respective K-mesh constant.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

%*** fix the blow-up factor for the K-mesh constant,
%*** where we assume C(Mesh_0) = 1, i.e., the initial mesh is uniform
kappa = 2;   

%*** count number of boundary parts from input 
%*** nB will hold this number 
nB = 0;

for iter = 1 : (nargin - 1)
   
    if size(varargin{iter},2) == 2
        nB = nB + 1;
        nE_boundary(iter) = size(varargin{iter},1);
    else
        break;
    end

end

%*** check the correct number of input parameters
if ~( (nargin == (nB+1)) || (nargin == (2*nB+1)) )
    error('refineBoundaryMesh: Wrong number of input arguments!');
end

%*** check the correct number of output parameters
if ~( (nargout == (nB+1)) || (nargout == (2*nB+1)) )
    error('refineBoundaryMesh: Wrong number of output arguments!');
end

%*** check, if user asks for father2son fields in output
if nargout == (2*nB+1)
    output_father2son = true;
else
    output_father2son = false;
end

%*** obtain set of all elements of the boundary partition
elements = cat(1,varargin{1 : nB});

%*** indices of a boundary part w.r.t. entire field elements
ptr_boundary = cumsum([0,nE_boundary]);

%*** 1. determine whether uniform or adaptive mesh-refinement
%*** 2. in case of adaptive mesh-refinement compute vector marked
%***    of marked elements w.r.t. entire field elements
if (nB+1) == nargin
    refinement = 'uniform';
else
    refinement = 'adaptive';
    marked = zeros(0,1); % marked elements w.r.t. entire field elements
 
    for iter = 1 : nB
        marked = [marked; varargin{iter + nB} + ptr_boundary(iter)];      
    end

end

nC = size(coordinates,1); % number of coordinates
nE = size(elements,1);    % number of elements

if strcmp(refinement,'adaptive')

    %*** if element Ej is marked and if its neighbour Ek satisfies 
    %*** hk >= kappa*hj, we (recursively) mark Ek for refinement as well

    %*** marked elements Ej will be refined, i.e., flag(j) = 1
    flag = zeros(nE,1);
    flag(marked) = 1;
    
    %*** determine neighbouring elements
    node2element = zeros(nC,2);
    node2element(elements(:,1),2) = (1:nE)';
    node2element(elements(:,2),1) = (1:nE)';
    element2neighbour = [ node2element(elements(:,1),1), ...
                          node2element(elements(:,2),2) ];

    %*** compute (squared) local mesh-size 
    h = sum((coordinates(elements(:,1),:)-coordinates(elements(:,2),:)).^2,2)';
    
    %*** the formal recursion is avoided by sorting elements by mesh-size
    [tmp,sorted_elements] = sort(h);
    for j = sorted_elements
        if flag(j)
            neighbours = element2neighbour(j,:);
            neighbours = neighbours( find(neighbours) );
            flag( neighbours(h(neighbours) >= kappa*h(j)) ) = 1;
        end
    end

    %*** obtain vector of marked elements
    marked = find(flag);
    nM = length(marked);

    %*** compute and add new nodes
    coordinates = [coordinates;zeros(nM,2)];
    coordinates((1:nM)+nC,:) = ( coordinates(elements(marked,1),:) ...
                                 + coordinates(elements(marked,2),:) )*0.5;

    %*** refinement of mesh iterates over each boundary part
    for iter = 1:nB
        
        %*** determine which marked elements belong to boundary part
        idx = find( (ptr_boundary(iter) < marked) ...
                    & (marked <= ptr_boundary(iter+1)) );
        nM_boundary = length(idx);
 
        %*** allocate new elements 
        new_elements = [varargin{iter};zeros(nM_boundary,2)];

        %*** generate new elements
        new_elements((1:nM_boundary)+nE_boundary(iter),:) ...
            = [ nC + idx, elements(marked(idx),2) ];
        new_elements( marked(idx) - ptr_boundary(iter),2 ) = nC + idx;

        %*** add new_elements and father2son to output
        varargout{iter} = new_elements;        

        %*** compute father2son only if desired
        if output_father2son == true

            %*** generate father2son
            father2son = repmat((1:nE_boundary(iter))',1,2);
            father2son( marked(idx) - ptr_boundary(iter),2 ) ...
                = (1:nM_boundary)' + nE_boundary(iter);
        
            %*** add new_elements and father2son to output
            varargout{nB+iter} = father2son;
        end

    end
    
elseif strcmp(refinement,'uniform')

    %*** compute and add new nodes
    coordinates = [coordinates;zeros(nE,2)];
    coordinates((nC+1):end,:) = ( coordinates(elements(:,1),:) ...
                                  + coordinates(elements(:,2),:) )*0.5;
    
    %*** uniform refinement of mesh iterates over each boundary part
    for iter = 1:nB

        %*** generate new elements
        idx = (ptr_boundary(iter)+1):ptr_boundary(iter+1);
        varargout{iter} = [ varargin{iter}(:,1),nC + idx' ; ...
                            nC + idx', varargin{iter}(:,2) ];

        %*** compute father2son only if desired
        if output_father2son == true

            %*** build father2son
            varargout{nB+iter} ...
                = [(1:nE_boundary(iter))', ...
                   (1:nE_boundary(iter))' + nE_boundary(iter)];
        end

    end
end

