function plotArclengthS1(coordinates,elements,x,varargin)
%PLOTARCLENGTHS1   Plots S1-function.
%   PLOTARCLENGTHS1(COORDINATES,ELEMENTS,UDH [,UD] [,FIGURE]) plots a 
%   globally continuous and piecewise affine function, for instance an S1-
%   Galerkin approximation, over the arclength of a connected boundary 
%   piece. Optionally, an additional function, for instance the exact 
%   solution, is plotted in the same figure.
%
%   Let {E1,...,EN} be a partition of a connected boundary piece Gamma with
%   nodes z1,...,zn. We stress that n = N+1 for an open boundary, whereas 
%   n = N for a closed boundary. This function plots an S1-function uDh over
%   the arclength of Gamma. Optionally, an additional function uD is plotted
%   in the same figure for comparison.
%
%   The (n x 2)-matrix COORDINATES gives the coordinates for the nodes on 
%   Gamma. The (N x 2)-matrix ELEMENTS describes the elements on Gamma. The
%   (n x 1)-vector UDH contains the nodal values of uDh, i.e. uDh(zj) = 
%   UDH(j). The optional function handle UD provides an implementation of 
%   the function uD. It takes an (M x 2)-matrix of nodes X and returns an 
%   (M x 1)-vector Y of corresponding function values, i.e., Y(j) = 
%   UD(X(j,:)). An optional positive parameter FIGURE gives the number of 
%   the output figure. If the optional parameter FIGURE is zero, the current
%   figure is used, but cleared before. If the optional parameter FIGURE is
%   negative, the current figure is used, but not cleared before. This is 
%   necessary to plot into subfigures.
          
% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

nC = size(coordinates,1); % number of coordinates
nE = size(elements,1);    % number of elements

%*** check whether optional figure number is given
if nargin > 3 && isnumeric(varargin{1})
    fig = varargin{1};
elseif nargin > 4 && isnumeric(varargin{2})
    fig = varargin{2};
else
    fig = 0;
end
if fig > 0
    figure(fig);
elseif fig == 0
    clf;
end

%*** check whether optional function uD is given
if nargin > 3 && isa(varargin{1},'function_handle')
    uD = varargin{1};
elseif nargin > 4 && isa(varargin{2},'function_handle')
    uD = varargin{2};
else
    uD = false;
end

%*** coordinatess(1,:) ist the same after refinement. therefore:
%*** find element E which has node 1 as left endpoint and switch
%*** with first element
tmp = find(elements(:,1)==1);
swap = elements(1,:);
elements(1,:) = elements(tmp,:);
elements(tmp,:) = swap;

%*** determine neighbour Ek of Ej with respect to second node of Ej
node2element = zeros(nC,1);
node2element(elements(:,1)) = (1:nE)';
element2neighbour = node2element(elements(:,2));

%*** re-order elements such that Ej+1 is neighbour of Ej
idx = ones(nE,1);
for j = 2:nE
    idx(j) = element2neighbour(idx(j-1));
end
elements = elements(idx,:);

%*** compute local mesh-size
h = sqrt(sum((coordinates(elements(:,1),:)-coordinates(elements(:,2),:)).^2,2));

%*** compute arclength s.t. [s(j),s(j+1)] corresponds to element Ej
s = [0;cumsum(h)];

%*** compute integral mean of discrete solution
x = [x(elements(:,1));x(elements(nE,2))];

%*** plot normalized discrete solution over arclength
plot(s,x,'-')

%*** optionally, we plot the exact solution
if isa(uD,'function_handle')
    %*** evaluation points for uD per element w.r.t. reference element [0,1]
    eval = 0:0.01:1;
    nEval = length(eval);
    
    %*** compute all evaluation points for uD
    a = coordinates(elements(:,1),:);
    b = coordinates(elements(:,2),:);
    sx = reshape(a,2*nE,1)*(1-eval) + reshape(b,2*nE,1)*eval;
    sx = reshape(sx',nEval*nE,2);
    
    %*** evaluate uD at all evaluation points and add an appropriate constant
    uD_sx = uD(sx);
    uD_sx = uD_sx - min(uD_sx) + min(x);
    
    %*** build arclength parameters corresponding to evaluation points
    s_sx = repmat(eval',1,nE).*repmat(h',nEval,1) + repmat(s(1:nE)',nEval,1);
    s_sx = reshape(s_sx,nE*nEval,1);
    
    %*** plot exact solution into the same figure
    hold on
    plot(s_sx,uD_sx,'r--')
    hold off
    axis([0 max(s) min(x) max(x)])
end

