function plotArclengthP0(coordinates,elements,x,varargin)
%PLOTARCLENGTHP0   Plots P0-function.
%   PLOTARCLENGTHP0(COORDINATES,ELEMENTS,PHIH, [,PHI] [,FIGURE]) plots a 
%   piecewise constant function, for instance the P0-Galerkin approximation,
%   over the arclength of a connected boundary piece. Optionally, an 
%   additional function, for instance the exact solution, is plotted in the
%   same figure.
%
%   Let {E1,...,EN} be a partition of a connected boundary piece Gamma. This
%   function plots a piecewise constant function phih over the arclength of
%   Gamma. Optionally, an additional function phi is plotted in the same 
%   figure for comparison.
%
%   COORDINATES gives the coordinates for the nodes on Gamma. The (N x 2)-
%   matrix ELEMENTS describes the elements on Gamma. The (N x 1)-vector PHIH
%   contains the elementwise values of phih, i.e. phih|_{Ej} = PHIH(j). The
%   optional function handle PHI provides an implementation of the function
%   phi. Since we are thinking of phi as a normal derivative, the function 
%   takes an (M x 2)-matrix of nodes X and the respective boundary elements
%   given by the (M x 2)-matrices A and B. It then returns the (M x 1)-
%   vector Y of corresponding function values, i.e.
%
%      Y(j) := PHI(X(j,:),A(j,:),B(j,:))
%
%   where X(j,:) lies on the boundary element with nodes A(j,:) and B(j,:). 
%   An optional positive parameter FIGURE gives the number of the output 
%   figure. If the optional parameter FIGURE is zero, the current figure is
%   used, but cleared before. If the optional parameter FIGURE is negative,
%   the current figure is used, but not cleared before. This is necessary to
%   plot into subfigures.
 
% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

nC = size(coordinates,1); % number of coordinates
nE = size(elements,1);    % number of elements

%*** check whether optional figure number is given
if nargin > 3 && isnumeric(varargin{1})
    fig = varargin{1};
elseif nargin > 4 && isnumeric(varargin{2})
    fig = varargin{2};
else
    fig = 0;
end
if fig > 0
    figure(fig);
elseif fig == 0
    clf;
end

%*** check whether optional function phi is given
if nargin > 3 && isa(varargin{1},'function_handle')
    phi = varargin{1};
elseif nargin > 4 && isa(varargin{2},'function_handle')
    phi = varargin{2};
else
    phi = false;
end

%*** coordinates(1,:) is the same after refinement. therefore:
%*** find element E which has node 1 as left endpoint and switch
%*** with first element
tmp = find(elements(:,1)==1);
swap = elements(1,:);
elements(1,:) = elements(tmp,:);
elements(tmp,:) = swap;

%*** switch solution accordingly, since it is associated with elements
swap = x(1);
x(1) = x(tmp);
x(tmp) = swap;

%*** determine neighbour Ek of Ej with respect to second node of Ej
node2element = zeros(nC,1);
node2element(elements(:,1)) = (1:nE)';
element2neighbour = node2element(elements(:,2));

%*** re-order elements such that Ej+1 is neighbour of Ej
idx = ones(nE,1);
for j = 2:nE
    idx(j) = element2neighbour(idx(j-1));
end
elements = elements(idx,:);
x = x(idx);

%*** compute local mesh-size
h = sqrt(sum((coordinates(elements(:,1),:)-coordinates(elements(:,2),:)).^2,2));

%*** compute arclength s.t. [s(j),s(j+1)] corresponds to Ej
s = [0;cumsum(h)];

%*** plot discrete solution over arclength
xplot = reshape([s(1:nE),s(2:nE+1)]',1,2*nE);
yplot = reshape([x,x]',1,2*nE);
plot(xplot,yplot,'b-')

%*** optionally, we plot the exact solution
if isa(phi,'function_handle')
    %*** evaluation points for phi per element w.r.t. reference element [0,1]
    eval = 0:0.01:1;
    nEval = length(eval);
    
    %*** compute all evaluation points for phi
    a = coordinates(elements(:,1),:);
    b = coordinates(elements(:,2),:);
    sx = reshape(a,2*nE,1)*(1-eval) + reshape(b,2*nE,1)*eval;
    sx = reshape(sx',nEval*nE,2);
    
    %*** evaluate phi at all evaluation points
    a_sx = reshape(repmat(reshape(a,2*nE,1),1,nEval)',nE*nEval,2);
    b_sx = reshape(repmat(reshape(b,2*nE,1),1,nEval)',nE*nEval,2);
    phi_sx = phi(sx,a_sx,b_sx);

    %*** build arclength parameters corresponding to evaluation points
    s_sx = repmat(eval',1,nE).*repmat(h',nEval,1) + repmat(s(1:nE)',nEval,1);
    s_sx = reshape(s_sx,nE*nEval,1);
    
    %*** plot exact solution into the same figure
    hold on
    plot(s_sx,phi_sx,'r--')
    hold off
    axis([0 max(s) min(x) max(x)])
end

