function figure6_mixedVol(varargin)
%FIGURE6_MIXEDVOL   Creates Figure 6 from HILBERT paper.
%   FIGURE_MIXEDVOL([NVMAX]) provides an experimental comparison between 
%   uniform and adaptive mesh-refinement for the integral formulation of a 
%   mixed boundary value problem with non-vanishing volume force. Estimated 
%   error and error estimators are plotted versus the number of degrees of 
%   freedom, the memory consumption, and the CPU time.
%
%   We consider the mixed problem with volume force
%
%     -K*uN + V*phiD  = 1/2 + K*g - V*phi - N0*f
%      W*uN + K'*phiD = 1/2 - W*g - K'*phi - N1*f
%
%   on the boundary Gamma of the turned L-shape, which is equivalent to the
%   Laplace problem
% 
%      -Laplace(u) = f    in Omega
%                u = g    on GammaD
%            du/dn = phi  on GammaN
%
%   We prescribe the exact solution in polar coordinates
%
%      u(x) = ( (x-a)^2 + (y-b)^2 )^beta + r^{2/3} * cos(2theta/3)
%
%   for fixed parameters (a,b) = (.14 .14) and alpha = 0.9. Then, phi is the
%   normal derivative of u on Gamma and g is its trace. It is easily seen 
%   that phi and g are smooth, whereas the volume data f are singular at the
%   point (a,b). Moreover, uN is the trace of u on GammaN and phiD is the 
%   normal derivative of u on GammaD. uN is smooth whereas phiD is
%   non-smooth. 
%
%   For a given volume mesh with 6 triangles and the induced boundary 
%   partition into 8 line segments, we compute discrete solutions for both 
%   uniform and adaptive mesh-refinement, as long as the generated volume 
%   meshes T have less than NVMAX elements. If NVMAX is not specified as 
%   input input of the function, we choose NVMAX = 5000. We plot the 
%   (h-h/2)-error estimators eta and mu-tilde as well as the boundary and 
%   volume oscillations. All quantities are plotted over the number of 
%   boundary elements, the memory consumption, and over the computational 
%   time.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

if nargin > 0
    NVMAX = varargin{1};
else
    NVMAX = 5000;
end

%*** add hilberttools directory
addpath('../')

%*** load L-shaped domain
addpath('fig2and6/')
vertices0 = load('coordinates.dat')/4;
triangles0 = load('triangles.dat');

%*** split Gamma into Dirichlet and Neumann boundary
dirichlet0 = [6 5; 5 8];
neumann0 = [1 2;2 3;3 6;8 7;7 4;4 1];

%*** rotate domain Omega so that exact solution is symmetric
alpha = 3*pi/4;
vertices0 = vertices0*[cos(alpha) -sin(alpha);sin(alpha) cos(alpha)]';

%*** rearrange indices such that Neumann nodes are first
[vertices0,triangles0,coordinates0,neumann0,dirichlet0] ...
    = buildSortedMesh(vertices0,triangles0,neumann0,dirichlet0);

%*** initialize Dirichlet data
uDh = zeros(size(coordinates0,1),1);
uDh(unique(dirichlet0)) = g(coordinates0(unique(dirichlet0),:));

%*** start uniform computation
counter = 1;
coordinates = coordinates0;
dirichlet = dirichlet0;
neumann = neumann0;
vertices = vertices0;
triangles = triangles0;

while 1

    %*** start time-measurement for uniform mesh-refinement
    time_ = cputime;
    
    coordinates = coordinates0;
    dirichlet = dirichlet0;
    neumann = neumann0;
    vertices = vertices0;
    triangles = triangles0;
    for j=1:counter-1
        %*** refine mesh uniformly
        [vertices,triangles,neumann_new,dirichlet,~, ...
            father2neumann,~] = refineMesh(vertices, ...
            triangles,neumann,dirichlet);
        
        %*** rearrange indices such that Neumann nodes are first
        [vertices,triangles,coordinates,neumann_new,dirichlet] ...
            = buildSortedMesh(vertices,triangles,neumann_new,dirichlet);
        
        %*** prolongate neumann data to the fine mesh
        uDh(neumann_new(father2neumann(:,1),2)) = 0.5*sum(uDh(neumann),2);
        uDh(neumann_new(father2neumann(:,1),1)) = uDh(neumann(:,1));
        uDh(neumann_new(father2neumann(:,2),2)) = uDh(neumann(:,2));
        uDh(unique(dirichlet)) = g(coordinates(unique(dirichlet),:));
        
        neumann = neumann_new;
    end
    
    fprintf('uniform --- number of elements (boundary): N = %d\r',...
            size(dirichlet,1) + size(neumann,1))
    fprintf('uniform --- number of elements (domain): N = %d\r',...
            size(triangles,1))

    %*** compute coarse-mesh solution
    elements = [dirichlet;neumann];
    V = buildV(coordinates,elements);
    K = buildK(coordinates,elements);
    W = buildW(coordinates,elements);
    
    %*** compute degrees of freedom for coarse mesh
    nC = size(coordinates,1);
    nD = size(dirichlet,1);
    freeNeumann = setdiff(1:nC,unique(dirichlet));
    freeDirichlet = 1:nD;
    
    %*** discretize data and compute corresponding data oscillations
    [oscD,oscN,~,phiNh] ...
        = computeOscMixed(coordinates,dirichlet,neumann,@g,@phi);
    [oscV,fh] = computeOscVolume(vertices,triangles,@f);
    
    %*** compute right hand side and shrink it
    [b1,b2] = buildMixedVolRHS(coordinates,dirichlet,neumann,uDh,phiNh, ...
        vertices,triangles,fh,V,K,W);
    b1 = b1(freeNeumann);
    b2 = b2(freeDirichlet);
    
    % shrink integral operators
    V = V(freeDirichlet,freeDirichlet);
    K = K(freeDirichlet,freeNeumann);
    W = W(freeNeumann,freeNeumann);
    
    %*** compute Galerkin solution for coarse mesh
    x = [ W K' ; -K V ] \ [ b1 ; b2 ];
    
    %*** compute solution vectors w.r.t. P0(GammaD) and S1(GammaN)
    nN = length(freeNeumann);
    dim = length(unique(neumann));
    xN = zeros(dim,1);
    xN(freeNeumann) = x(1:nN);
    xD = x((1:nD) + nN);
    
    %*** stop time-measurement for uniform mesh-refinement
    time_ = cputime - time_;
    
    %*** memory consumption
    s = whos('coordinates','elements','vertices','volumes','V','K','W',...
             'b1','b2','xN','xD');
    memory_ = 0;
    for k=1:size(s,1)
    	memory_ = memory_ + s(k).bytes;
    end
    %*** free obsolete memory
    clear V K W b1 b2
    
    %*** refine mesh uniformly
    [coordinates_fine,dirichlet_fine,neumann_fine,...
        father2dirichlet,father2neumann] ...
        = refineBoundaryMesh(coordinates,dirichlet,neumann);
    
    %*** rearrange indices such that Neumann nodes are first
    [coordinates_fine,neumann_fine,dirichlet_fine] ...
        = buildSortedMesh(coordinates_fine,neumann_fine,dirichlet_fine);
    
    %*** discretize data and compute corresponding data oscillations for
    %*** fine mesh
    [~,~,uDh_fine,phiNh_fine] ...
        = computeOscMixed(coordinates_fine,dirichlet_fine,neumann_fine, ...
        father2neumann,neumann,uDh,@g,@phi);
    
    %*** compute integral operators for fine mesh
    elements_fine = [dirichlet_fine;neumann_fine];
    V_fine = buildV(coordinates_fine,elements_fine);
    K_fine = buildK(coordinates_fine,elements_fine);
    W_fine = buildW(coordinates_fine,elements_fine);
    
    %*** compute degrees of freedom for fine mesh
    nC_fine = size(coordinates_fine,1);
    nD_fine = size(dirichlet_fine,1);
    freeNeumann_fine = setdiff(1:nC_fine,unique(dirichlet_fine));
    freeDirichlet_fine = 1:nD_fine;
    nodesNeumann_fine = unique(neumann_fine);
    nN_fine = length(freeNeumann_fine);
    
    % compute right-hand side for fine mesh and shrink it
    [b1_fine,b2_fine,Nell_fine] = buildMixedVolRHS(coordinates_fine, ...
        dirichlet_fine,neumann_fine,uDh_fine,phiNh_fine, ...
        vertices,triangles,fh,V_fine,K_fine,W_fine);
    b1_fine = b1_fine(freeNeumann_fine);
    b2_fine = b2_fine(freeDirichlet_fine);
    
    %*** shrink integral operators
    V_fine = V_fine(freeDirichlet_fine,freeDirichlet_fine);
    K_fine = K_fine(freeDirichlet_fine,freeNeumann_fine);
    W_fine = W_fine(nodesNeumann_fine,nodesNeumann_fine);
    
    %*** compute Galerkin solution for fine mesh
    x = [ W_fine(freeNeumann_fine,freeNeumann_fine) K_fine' ; ...
        -K_fine V_fine ] ...
        \ [ b1_fine ; b2_fine ];
    
    %*** compute coefficient vectors w.r.t. S1(GammaN) and P0(GammaD)
    dim_fine = length(unique(neumann_fine));
    xN_fine = zeros(dim_fine,1);
    xN_fine(freeNeumann_fine) = x(1:nN_fine);
    xD_fine = x((1:nD_fine) + nN_fine);
    
    %*** compute error estimators and error bounds
    etaD = computeEstSlpEta(father2dirichlet,V_fine,xD_fine,xD);
    etaN = computeEstHypEta(neumann_fine,neumann,father2neumann,W_fine, ...
        xN_fine,xN);
    etaD_tilde = computeEstSlpEtaTilde(father2dirichlet,V_fine,xD_fine);
    etaN_tilde = computeEstHypEtaTilde(neumann_fine,neumann,father2neumann, ...
        W_fine,xN_fine);
    muD = computeEstSlpMu(coordinates,dirichlet,father2dirichlet,xD_fine,xD);
    muN = computeEstHypMu(neumann_fine,neumann,father2neumann,xN_fine,xN);
    muD_tilde = computeEstSlpMuTilde(coordinates,dirichlet,father2dirichlet,...
        xD_fine);
    muN_tilde = computeEstHypMuTilde(neumann_fine,neumann,father2neumann, ...
        xN_fine);
    errD = computeErrNeumann(coordinates,dirichlet,xD,@phi);
    xNg = uDh;
    xNg(freeNeumann) = xNg(freeNeumann) + x(1:nN);
    errN = computeErrDirichlet(coordinates,neumann,xNg,@g);
    mu_V_tilde_D = computeEstSlpMuTilde(coordinates,dirichlet, ...
        father2dirichlet,Nell_fine(1:nD_fine));
    mu_V_tilde_N = computeEstSlpMuTilde(coordinates,neumann, ...
        father2neumann,Nell_fine(nD_fine+1:end));
        
    %*** free obsolete memory
    clear coordinates_fine elements_fine dirichlet_fine neumann_fine
    clear father2son father2dirichlet father2neumann
    clear V_fine K_fine W_fine b1_fine b2_fine x_fine x
    
    nE_unif(counter) = size(elements,1);
    nV_unif(counter) = size(triangles,1);
    time_unif(counter) = time_;
    memory_unif(counter) = memory_;
    eta_unif(counter) = sqrt(etaD + etaN);
    eta_tilde_unif(counter) = sqrt(etaD_tilde + etaN_tilde);
    mu_unif(counter) = sqrt(sum([muD; muN]));
    mu_tilde_unif(counter) = sqrt(sum([muD_tilde; muN_tilde]));
    mu_V_tilde_unif(counter) = sqrt(sum([mu_V_tilde_D; mu_V_tilde_N]));
    osc_unif(counter) = sqrt(sum([oscD; oscN]));
    oscV_unif(counter) = sqrt(sum(oscV));
    err_unif(counter) = sqrt(sum([errD; errN]));
    
    %*** iterate
    counter = counter+1;
    if(max(nV_unif) > NVMAX)
        break
    end
end

%*** adaptive
theta = 0.25;
vertices = vertices0;
triangles = triangles0;
neumann = neumann0;
dirichlet = dirichlet0;
counter = 1;

%*** rearrange indices such that Neumann nodes are first, extract coordinates
[vertices,triangles,coordinates,neumann,dirichlet] ...
    = buildSortedMesh(vertices,triangles,neumann,dirichlet);

%*** initialize Dirichlet data
uDh = zeros(size(coordinates,1),1);
uDh(unique(dirichlet)) = g(coordinates(unique(dirichlet),:));

%*** start time-measurement for adaptive mesh-refinement
time_ = cputime;

%*** Perform uniform refinement before starting the loop
%*** refine mesh uniformly
[coordinates_fine,dirichlet_fine,neumann_fine,father2dirichlet_fine, ...
    father2neumann_fine] = refineBoundaryMesh(coordinates,dirichlet,neumann);

%*** rearrange indices such that Neumann nodes are first
[coordinates_fine,neumann_fine,dirichlet_fine] = ...
    buildSortedMesh(coordinates_fine,neumann_fine,dirichlet_fine);

%*** discretize data and compute corresponding data oscillations for fine mesh
[oscD_fine,oscN_fine,uDh_fine,phiNh_fine] = ...
    computeOscMixed(coordinates_fine,dirichlet_fine,neumann_fine, ...
    father2neumann_fine,neumann,uDh,@g,@phi);
[oscV,fh] = computeOscVolume(vertices,triangles,@f);

while 1
    
    fprintf('adaptive --- number of elements (boundary): N = %d\r',...
            size(dirichlet,1) + size(neumann,1))
    fprintf('adaptive --- number of elements (domain): N = %d\r',...
            size(triangles,1))

    if counter > 1
        %*** start time-measurement for adaptive mesh-refinement
        time_ = cputime;
        
        %***
        old_neumann_fine = neumann_fine;
        old_father2neumann_fine = father2neumann_fine;
        
        [coordinates_fine,dirichlet_fine,neumann_fine,father2dirichlet_fine,...
            father2neumann_fine] ...
            = refineBoundaryMesh(coordinates,dirichlet,neumann);
        
        %*** rearrange nodes such that neumann nodes are first
        [coordinates_fine,neumann_fine,dirichlet_fine] = ...
            buildSortedMesh(coordinates_fine,neumann_fine,dirichlet_fine);
        
        %*** build old2fine array which links old fine mesh to new fine mesh
        old2fine = generateFather2Son(father2neumann_new, ...
            old_father2neumann_fine, ...
            father2neumann_fine);
        
        %*** discretize data, compute oscillations for fine mesh
        [oscD_fine,oscN_fine,uDh_fine,phiNh_fine] = ...
            computeOscMixed(coordinates_fine,dirichlet_fine,neumann_fine, ...
            old2fine,old_neumann_fine,uDh_fine,@g,@phi);
        [oscV,fh] = computeOscVolume(vertices,triangles,@f);
    end
    
    oscD = sum(oscD_fine(father2dirichlet_fine),2);
    oscN = sum(oscN_fine(father2neumann_fine),2);
    
    %*** compute integral operators and right hand side for fine mesh
    elements_fine = [dirichlet_fine;neumann_fine];
    V_fine = buildV(coordinates_fine,elements_fine);
    K_fine = buildK(coordinates_fine,elements_fine);
    W_fine = buildW(coordinates_fine,elements_fine);
    [b1_fine,b2_fine,Nell_fine] = buildMixedVolRHS(coordinates_fine, ...
        dirichlet_fine,neumann_fine,uDh_fine,phiNh_fine,vertices,triangles, ...
        fh,V_fine,K_fine,W_fine);
    
    %*** compute degrees of freedom for fine mesh
    nC_fine = size(coordinates_fine,1);
    nD_fine = size(dirichlet_fine,1);
    freeNeumann_fine = setdiff(1:nC_fine,unique(dirichlet_fine));
    freeDirichlet_fine = 1:nD_fine;
    nN_fine = length(freeNeumann_fine);
    
    %*** compute fine-mesh Galerkin solution
    V_fine = V_fine(freeDirichlet_fine,freeDirichlet_fine);
    K_fine = K_fine(freeDirichlet_fine,freeNeumann_fine);
    nodesNeumann = unique(neumann_fine);
    W_fine = W_fine(nodesNeumann,nodesNeumann);
    x = [ W_fine(freeNeumann_fine,freeNeumann_fine) K_fine' ;  ...
        -K_fine V_fine ] ...
        \ [ b1_fine(freeNeumann_fine) ; b2_fine(freeDirichlet_fine) ];
    
    %*** compute coefficient vectors w.r.t. S1(GammaN) and P0(GammaD)
    dim_fine = length(unique(neumann_fine));
    xN_fine = zeros(dim_fine,1);
    xN_fine(freeNeumann_fine) = x(1:nN_fine); %** dof on Neumann boundary
    xD_fine = x((1:nD_fine) + nN_fine);       %** dof on Dirichlet boundary
    
    %*** compute error estimates
    muD_tilde = computeEstSlpMuTilde(coordinates,dirichlet, ...
        father2dirichlet_fine,xD_fine);
    muN_tilde = computeEstHypMuTilde(neumann_fine,neumann, ...
        father2neumann_fine,xN_fine);
    mu_V_tilde_D = computeEstSlpMuTilde(coordinates,dirichlet, ...
        father2dirichlet_fine,Nell_fine(1:nD_fine));
    mu_V_tilde_N = computeEstSlpMuTilde(coordinates,neumann, ...
        father2neumann_fine,Nell_fine(nD_fine+1:end));
    
    %*** mark elements for refinement
    [marked_dirichlet,marked_neumann,marked_triangles] ...
        = markElements(theta,muD_tilde + oscD + mu_V_tilde_D, ...
        muN_tilde + oscN + mu_V_tilde_N,oscV);
    
    %*** generate new mesh
    [vertices_new,triangles_new,dirichlet_new,neumann_new,...
        ~,~,father2neumann_new] ...
        = refineMesh(vertices,triangles,dirichlet,neumann,marked_triangles, ...
        marked_dirichlet,marked_neumann);
    
    %*** rearrange indices such that neumann nodes are first
    [vertices_new,triangles_new,coordinates_new,neumann_new,dirichlet_new] = ...
        buildSortedMesh(vertices_new,triangles_new,neumann_new,dirichlet_new);
    
    %*** stop time-measurement for adaptive mesh-refinement
    time_ = cputime - time_;
    if counter>1
        time_ = time_adap(counter-1)+time_;
    end
    
    %*** memory consumption
    s = whos('coordinates_fine','dirichlet_fine','neumann_fine','father2dirichlet', ...
        'father2neumann','coordinates','dirichlet','neumann','elements_fine', ...
	'V_fine','K_fine','W_fine','b1_fine','b2_fine','freeNeumann_fine', ...
	'freeDirichlet_fine','nodesNeumann','x','oscD_fine','oscN_fine', ...
	'muD_tilde','muN_tilde','mu1_tilde_D','mu1_tilde_N','marked_dirichlet', ...
	'marked_neumann','marked_volumes','vertices_new','volumes_new','dirichlet_new', ...
	'neumann_new','father2volumes','father2neumann_new','father2dirichlet_new', ...
	'elements_new','coordinates_new');
    memory_ = 0;
    for k=1:size(s,1)
    	memory_ = memory_ + s(k).bytes;
    end
    
    %*** compute coarse mesh solution
    %*** compute degrees of freedom for coarse mesh
    nC = size(coordinates,1);
    nD = size(dirichlet,1);
    freeNeumann = setdiff(1:nC,unique(dirichlet));
    freeDirichlet = 1:nD;
    
    %*** compute integral operators and right hand side for coarse mesh
    elements = [dirichlet;neumann];
    father2son = [father2dirichlet_fine ; father2neumann_fine + nD_fine];
    V = buildV(coordinates,elements);
    K = buildK(coordinates,elements);
    W = buildW(coordinates,elements);
    
    %*** build index field k = coarse2fine (j) such that j-th node of coarse
    %*** mesh coincides with k-th node of fine mesh
    nC = size(coordinates,1);
    coarse2fine = zeros(nC,1);
    coarse2fine(elements) = [ elements_fine(father2son(:,1),1) ...
        elements_fine(father2son(:,2),2) ];
    
    %*** build index field k = element2midpoint(j) such that midpoint of j-th
    %*** element of coarse mesh is k-th node of fine mesh
    element2midpoint = elements_fine(father2son(:,1),2);
    
    %*** build index field [i,j] = node2element(k) such that
    %*** intersection(Ei,Ej) = zk
    nE_ = size(elements,1);
    node2element = zeros(nC,2);
    node2element(elements(:,1),2) = (1:nE_)';
    node2element(elements(:,2),1) = (1:nE_)';
    
    %*** obtain coarse-mesh RHS from fine-mesh RHS
    b1 = b1_fine(coarse2fine) ...
        + 0.5*sum(b1_fine(element2midpoint(node2element)),2);
    b2 = b2_fine(father2son(:,1))+b2_fine(father2son(:,2));
    
    %*** compute Galerkin solution w.r.t. coarse mesh for error estimation
    V = V(freeDirichlet,freeDirichlet);
    K = K(freeDirichlet,freeNeumann);
    W = W(freeNeumann,freeNeumann);
    x = [ W K' ; -K V ] \ [ b1(freeNeumann) ; b2(freeDirichlet) ];
    
    %*** compute solution vectors w.r.t. P0(GammaD) und S1(GammaN)
    nN = length(freeNeumann);
    dim = length(unique(neumann));
    xN = zeros(dim,1);
    xN(freeNeumann) = x(1:nN);     %** dof on Neumann boundary
    xD = x((1:nD) + nN);           %** dof on Dirichlet boundary
    
    %*** compute mu estimators for GammaD and GammaN
    muD = computeEstSlpMu(coordinates,dirichlet,father2dirichlet_fine,...
                          xD_fine,xD);
    muN = computeEstHypMu(neumann_fine,neumann,father2neumann_fine,xN_fine,xN);
    
    %*** compute tilde-eta estimators for GammaD and GammaN
    etaD_tilde = computeEstSlpEtaTilde(father2dirichlet_fine,V_fine,xD_fine);
    etaN_tilde = computeEstHypEtaTilde(neumann_fine,neumann,...
        father2neumann_fine,W_fine,xN_fine);
    
    %*** compute eta estimators for GammaD and GammaN
    etaD = computeEstSlpEta(father2dirichlet_fine,V_fine,xD_fine,xD);
    etaN = computeEstHypEta(neumann_fine,neumann,father2neumann_fine,W_fine, ...
                            xN_fine,xN);
    
    %*** compute reliable error bound for GammaD and GammaN
    errD = computeErrNeumann(coordinates,dirichlet,xD,@phi);
    xNg = uDh_fine(coarse2fine);
    
    xNg(freeNeumann) = xNg(freeNeumann) + x(1:nN);
    errN = computeErrDirichlet(coordinates,neumann,xNg,@g);
    
    %*** store computed data
    nE_adap(counter) = size(elements,1);
    nV_adap(counter) = size(triangles,1);
    time_adap(counter) = time_;
    memory_adap(counter) = memory_;
    eta_adap(counter) = sqrt(etaD + etaN);
    eta_tilde_adap(counter) = sqrt(etaD_tilde + etaN_tilde);
    mu_adap(counter) = sqrt(sum([muD ; muN]));
    mu_tilde_adap(counter) = sqrt(sum([muD_tilde; muN_tilde]));
    mu_V_tilde_adap(counter) = sqrt(sum([mu_V_tilde_D; mu_V_tilde_N]));
    osc_adap(counter) = sqrt(sum([oscD; oscN]));
    oscV_adap(counter) = sqrt(sum(oscV));
    err_adap(counter) = sqrt(sum([errD ; errN]));
        
    %*** increase counter and iterate
    counter = counter + 1;
    coordinates = coordinates_new;
    dirichlet = dirichlet_new;
    neumann = neumann_new;
    vertices = vertices_new;
    triangles = triangles_new;
    if(max(nV_adap) > NVMAX)
        break
    end
end

%*** create plots
art = {'LineWidth',2,'MarkerSize',7};
style = {'r+-.','gs-.','bo-.','c^-.','mx-.','kd-.','r+-','gs-','bo-',...
         'c^-','mx-','kd-'};
txt = {'error (unif)','eta (unif)','tilde-mu (unif)','osc (unif)',...
       'oscV (unif)','tilde-muV (unif)', ...
       'error (adap)','eta (adap)','tilde-mu (adap)','osc (adap)',...
       'oscV (adap)','tilde-muV (adap)'};

%*** adaptive: plot only every second step for better overview
stop = size(nE_adap,2);
nE_adap = nE_adap(1:2:stop);
time_adap = time_adap(1:2:stop);
memory_adap = memory_adap(1:2:stop);
err_adap = err_adap(1:2:stop);
mu_tilde_adap = mu_tilde_adap(1:2:stop);
osc_adap = osc_adap(1:2:stop);
oscV_adap = oscV_adap(1:2:stop);
mu_V_tilde_adap = mu_V_tilde_adap(1:2:stop);
eta_adap = eta_adap(1:2:stop);

%*** plot versus number of elements
h = figure(1);
clf;
pos = get(h,'Position');
pos(1) = 0;
pos(2) = 0;
pos(3) = 1.5*pos(3);
pos(4) = 1.5*pos(4);
set(h,'Position',pos)
loglog(nE_unif,err_unif+osc_unif+oscV_unif,style{1},art{:})
hold on
loglog(nE_unif,eta_unif,style{2},art{:})
loglog(nE_unif,mu_tilde_unif,style{3},art{:})
loglog(nE_unif,osc_unif,style{4},art{:})
loglog(nE_unif,oscV_unif,style{5},art{:})
loglog(nE_unif,mu_V_tilde_unif,style{6},art{:})
loglog(nE_adap,err_adap+osc_adap+oscV_adap,style{7},art{:})
loglog(nE_adap,eta_adap,style{8},art{:})
loglog(nE_adap,mu_tilde_adap,style{9},art{:})
loglog(nE_adap,osc_adap,style{10},art{:})
loglog(nE_adap,oscV_adap,style{11},art{:})
loglog(nE_adap,mu_V_tilde_adap,style{12},art{:})
loglog(nE_adap,nE_adap.^(-3/2),'k--',art{:})
loglog(nE_unif,nE_unif.^(-4/7),'k--',art{:})
axis([5 5*10^(3) 10^(-5.5) 10^(0.5)]);
myLegend(0.9,-2.5,-5.5,style,txt)
ylabel('error and estimators')
xlabel('number of elements')
text(10^(3.1),10^(-4.4),'-3/2')
text(10^(2.5),10^(-1.2),'-4/7')
print -r600 -depsc2 fig6_mixedVol_dofs.eps

%*** plot versus computation time
h = figure(2);
clf;
pos = get(h,'Position');
pos(1) = 0;
pos(2) = 0;
pos(3) = 1.5*pos(3);
pos(4) = 1.5*pos(4);
set(h,'Position',pos)
loglog(time_unif,err_unif+osc_unif+oscV_unif,style{1},art{:})
hold on
loglog(time_unif,eta_unif,style{2},art{:})
loglog(time_unif,mu_tilde_unif,style{3},art{:})
loglog(time_unif,osc_unif,style{4},art{:})
loglog(time_unif,oscV_unif,style{5},art{:})
loglog(time_unif,mu_V_tilde_unif,style{6},art{:})
loglog(time_adap,err_adap+osc_adap+oscV_adap,style{7},art{:})
loglog(time_adap,eta_adap,style{8},art{:})
loglog(time_adap,mu_tilde_adap,style{9},art{:})
loglog(time_adap,osc_adap,style{10},art{:})
loglog(time_adap,oscV_adap,style{11},art{:})
loglog(time_adap,mu_V_tilde_adap,style{12},art{:})
xlabel('computational time');
print -r600 -depsc2 fig6_mixedVol_time.eps

%*** plot versus memory consumption
h = figure(3);
clf;
pos = get(h,'Position');
pos(1) = 0;
pos(2) = 0;
pos(3) = 1.5*pos(3);
pos(4) = 1.5*pos(4);
set(h,'Position',pos)
loglog(memory_unif/1048576,err_unif+osc_unif+oscV_unif,style{1},art{:})
hold on
loglog(memory_unif/1048576,eta_unif,style{2},art{:})
loglog(memory_unif/1048576,mu_tilde_unif,style{3},art{:})
loglog(memory_unif/1048576,osc_unif,style{4},art{:})
loglog(memory_unif/1048576,oscV_unif,style{5},art{:})
loglog(memory_unif/1048576,mu_V_tilde_unif,style{6},art{:})
loglog(memory_adap/1048576,err_adap+osc_adap+oscV_adap,style{7},art{:})
loglog(memory_adap/1048576,eta_adap,style{8},art{:})
loglog(memory_adap/1048576,mu_tilde_adap,style{9},art{:})
loglog(memory_adap/1048576,osc_adap,style{10},art{:})
loglog(memory_adap/1048576,oscV_adap,style{11},art{:})
loglog(memory_adap/1048576,mu_V_tilde_adap,style{12},art{:})
xlabel('memory [MB]');
print -r600 -depsc2 fig6_mixedVol_mem.eps
