function figure5_hypsing(varargin)
%FIGURE5_HYPSING   Creates Figure 5 from HILBERT paper.
%   FIGURE5_HYPSING([NMAX]) provides an experimental comparison between 
%   uniform and adaptive mesh-refinement for the hypersingular integral 
%   equation on a rotated L-shaped boundary, where the solution as well as 
%   the Neumann data have singularities of different order. Estimated error
%   and error estimators are plotted versus the number of degrees of 
%   freedom, the memory consumption, and CPU time.
%
%   We consider the hypersingular integral equation 
%
%      W*u = (1/2-K')*phi
%
%   on an L-shaped boundary Gamma, which is equivalent to the Laplace 
%   problem
% 
%      -Laplace(u) = 0 in Omega with du/dn = phi on Gamma
%
%   For given delta>0, we define a function in polar coordinates
%
%      v_delta(r,theta) = r^delta * cos(delta*theta)
%
%   We prescribe the exact solution of the Laplace equation by
%
%      u(x) = v_{2/3}(x) + v_{7/8}(x-z)
%
%   where z is the uppermost corner of Gamma. Then, the solution u of the 
%   hypersingular integral equation is just the trace of the solution u of 
%   the Laplace problem. 
%
%   For a given initial partition E0 with 8 uniform elements, we compute 
%   discrete solutions for both, uniform and adaptive mesh-refinement, as 
%   long as the generated meshes E have less than NMAX elements. If NMAX is
%   not specified as input of the function, we choose NMAX = 1000. We plot 
%   the (h-h/2)-error estimators eta, eta-tilde, mu, and mu-tilde as well as
%   a reliable upper bound for the error. All quantities are plotted over 
%   the number of elements, the memory consumption, and over the 
%   computational time.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

if nargin > 0
    NMAX = varargin{1};
else
    NMAX = 1000;
end

%*** add hilberttools directory
addpath('../');

%*** load L-shaped domain
addpath('fig3to5/');
coordinates0 = load('coordinates.dat');
elements0 = load('elements.dat');

%*** rotate domain Omega so that exact solution is symmetric
alpha = 3*pi/4;
coordinates0 = coordinates0*[cos(alpha) -sin(alpha);sin(alpha) cos(alpha)]';

%*** shrink domain to ensure ellipticity of V by diam(Omega)<1
coordinates0 = coordinates0/4;

%*** uniform
elements = elements0;
counter=1;
while 1

    %*** start time-measurement for uniform mesh-refinement
    time_ = cputime;
    
    %*** build mesh by successive uniform refinements of initial mesh
    coordinates = coordinates0;
    elements = elements0;

    for j = 1:counter-1
        [coordinates,elements] = refineBoundaryMesh(coordinates,elements);
    end
    
    fprintf('uniform --- number of elements: N = %d\r',size(elements,1))
    
    %*** discretize Neumann data
    [~,phih] = computeOscNeumann(coordinates,elements,@phi);

    %*** compute coarse-mesh solution
    W = buildW(coordinates,elements) ...
        + buildHypsingStabilization(coordinates,elements);
    
    b = buildHypsingRHS(coordinates,elements,phih);
    x = W\b;
    
    %*** stop time-measurement for uniform mesh-refinement
    time_ = cputime - time_;

    %*** memory consumption
    s = whos('coordinates','elements','W','b','x');
    memory_ = 0;
    for k=1:size(s,1)
    	memory_ = memory_ + s(k).bytes;
    end

    %*** free obsolete memory
    clear W b phih s
    
    %*** generate uniformly refined mesh
    [coordinates_fine,elements_fine,father2son] ...
        = refineBoundaryMesh(coordinates,elements);
    
    %*** discretize Neumann data
    [~,phih_fine] = computeOscNeumann(coordinates_fine,elements_fine,@phi);

    %*** compute fine-mesh solution to estimate error
    W_fine = buildW(coordinates_fine,elements_fine) ...
             + buildHypsingStabilization(coordinates_fine,elements_fine);
    b_fine = buildHypsingRHS(coordinates_fine,elements_fine,phih_fine);
    x_fine = W_fine\b_fine;
    
    %*** compute (h-h/2)-type estimators
    eta_ = computeEstHypEta(elements_fine,elements,father2son, ...
                            W_fine,x_fine,x);
    eta_tilde_ = computeEstHypEtaTilde(elements_fine,elements,father2son, ...
                                       W_fine,x_fine);
    mu_ = computeEstHypMu(elements_fine,elements,father2son,x_fine,x);
    mu_tilde_ = computeEstHypMuTilde(elements_fine,elements,father2son, ...
                                     x_fine);

    %*** compute error bound
    err_ = computeErrDirichlet(coordinates_fine,elements_fine,x_fine,@g);

    %*** free obsolete memory
    clear coordinates_fine elements_fine father2son
    clear W_fine b_fine x_fine phih_fine x
    
    %*** compute Neumann data oscillation
    osc_ = computeOscNeumann(coordinates,elements,@phi);
    
    %*** store computed data
    N_unif(counter) = size(elements,1);
    time_unif(counter) = time_;
    memory_unif(counter) = memory_;
    eta_unif(counter) = sqrt(eta_);
    eta_tilde_unif(counter) = sqrt(eta_tilde_);
    mu_unif(counter) = sqrt(sum(mu_));
    mu_tilde_unif(counter) = sqrt(sum(mu_tilde_));
    err_unif(counter) = sqrt(sum(err_));
    osc_unif(counter) = sqrt(sum(osc_));
        
    %*** increase counter and iterate
    counter = counter+1;
    if (max(N_unif) > NMAX)
    	break
    end
end

%*** adaptive
theta = 0.25;
coordinates = coordinates0;
elements = elements0;
counter=1;
while 1

    fprintf('adaptive --- number of elements: N = %d\r',size(elements,1))

    %*** start time-measurement for adaptive mesh-refinement
    time_ = cputime;
    
    %*** generate uniformly refined mesh
    [coordinates_fine,elements_fine,father2son] ...
        = refineBoundaryMesh(coordinates,elements);
    
    %*** discretize Neumann data and compute data oscillations
    [osc_fine,phih_fine] = computeOscNeumann(coordinates_fine, ...
                                             elements_fine,@phi);
    osc_ = osc_fine(father2son(:,1)) + osc_fine(father2son(:,2));

    %*** compute fine-mesh solution
    W_fine = buildW(coordinates_fine,elements_fine) ...
             + buildHypsingStabilization(coordinates_fine,elements_fine);
    b_fine = buildHypsingRHS(coordinates_fine,elements_fine,phih_fine);
    x_fine = W_fine\b_fine;
    
    %*** compute (h-h/2)-error estimator tilde-mu
    mu_tilde_ = computeEstHypMuTilde(elements_fine,elements,father2son,x_fine);

    %*** mark elements for refinement
    marked = markElements(theta,1/4,mu_tilde_+osc_);

    %*** generate new mesh
    [coordinates_new,elements_new] ...
        = refineBoundaryMesh(coordinates,elements,marked);
    
    %*** stop time-measurement for adaptive mesh-refinement
    time_ = cputime - time_;
    if counter>1
        time_ = time_adap(counter-1)+time_;
    end

    %*** memory consumption
    s = whos('coordinates_fine','elements_fine', ...
        'coordinates_new','elements_new', ...
	'coordinates','elements', 'marked_elements', ...
	'W_fine','b_fine','mu_tilde','osc_fine','osc_','x_fine');
    memory_ = 0;
    for k=1:size(s,1)
    	memory_ = memory_ + s(k).bytes;
    end
    
    %*** build index field k = coarse2fine (j) such that j-th node of coarse
    %*** mesh coincides with k-th nodes of fine mesh
    nC = size(coordinates,1);
    coarse2fine = zeros(nC,1);
    coarse2fine(elements) = [ elements_fine(father2son(:,1),1) ...
                              elements_fine(father2son(:,2),2) ];
                          
    %*** build index field k = element2midpoint(j) such that midpoint of j-th
    %*** element of coarse mesh is k-th node of fine mesh
    element2midpoint = elements_fine(father2son(:,1),2);

    %*** build index field [i,j] = node2element(k) such that
    %*** intersection(Ei,Ej) = zk
    nE_ = size(elements,1);
    node2element = zeros(nC,2);
    node2element(elements(:,1),2) = (1:nE_)';
    node2element(elements(:,2),1) = (1:nE_)';
 
    %*** obtain coarse-mesh RHS from fine-mesh RHS
    b = b_fine(coarse2fine) + 0.5*sum(b_fine(element2midpoint(node2element)),2);
 
    %*** compute coarse-mesh solution for error estimation
    W = buildW(coordinates,elements) ...
        + buildHypsingStabilization(coordinates,elements);
    x = W\b;

    %*** free obsolete memory
    clear W b
    clear idx coarse2fine element2midpoint node2element 

    %*** compute (h-h/2)-estimators eta, tilde-eta, and mu
    eta_ = computeEstHypEta(elements_fine,elements,father2son,W_fine,x_fine,x);
    eta_tilde_ = computeEstHypEtaTilde(elements_fine,elements,father2son, ...
                                       W_fine,x_fine);
    mu_ = computeEstHypMu(elements_fine,elements,father2son,x_fine,x);
    err_ = computeErrDirichlet(coordinates_fine,elements_fine,x_fine,@g);

    %*** free obsolete memory
    clear coordinates_fine elements_fine father2son
    clear W_fine b_fine x_fine x

    %*** store computed data
    N_adap(counter) = size(elements,1);
    time_adap(counter) = time_;
    memory_adap(counter) = memory_;
    eta_adap(counter) = sqrt(eta_);
    eta_tilde_adap(counter) = sqrt(eta_tilde_);
    mu_adap(counter) = sqrt(sum(mu_));
    mu_tilde_adap(counter) = sqrt(sum(mu_tilde_));
    osc_adap(counter) = sqrt(sum(osc_));
    err_adap(counter) = sqrt(sum(err_));
    
    %*** increase counter and iterate
    counter = counter + 1;
    coordinates = coordinates_new;
    elements = elements_new;
    if max(N_adap) > NMAX
    	break
    end    
end

%*** create plots
art = {'LineWidth',2,'MarkerSize',7};
style = {'r+-.','gs-.','bo-.','c^-.', ...
         'r+-','gs-','bo-','c^-'};
txt = {'error (unif)','eta (unif)','tilde-mu (unif)','osc (unif)', ...
       'error (adap)','eta (adap)','tilde-mu (adap)','osc (adap)'};

%*** adaptive: plot only every second step for better overview
stop = size(N_adap,2);
N_adap = N_adap(1:2:stop);
time_adap = time_adap(1:2:stop);
memory_adap = memory_adap(1:2:stop);
eta_adap = eta_adap(1:2:stop);
err_adap = err_adap(1:2:stop);
mu_tilde_adap = mu_tilde_adap(1:2:stop);
osc_adap = osc_adap(1:2:stop);

%*** plot versus number of elements
h = figure(1);
clf;
pos = get(h,'Position');
pos(1) = 0;
pos(2) = 0;
pos(3) = 1.5*pos(3);
pos(4) = 1.5*pos(4);
set(h,'Position',pos)

loglog(N_unif,err_unif+osc_unif,style{1},art{:})
hold on
loglog(N_unif,eta_unif,style{2},art{:})
loglog(N_unif,mu_tilde_unif,style{3},art{:})
loglog(N_unif,osc_unif,style{4},art{:})
loglog(N_unif,2.5*N_unif.^(-2/3),'k--',art{:})

loglog(N_adap,err_adap+osc_adap,style{5},art{:})
hold on
loglog(N_adap,eta_adap,style{6},art{:})
loglog(N_adap,mu_tilde_adap,style{7},art{:})
loglog(N_adap,osc_adap,style{8},art{:})
loglog(N_unif,2.3*N_unif.^(-3/2),'k--',art{:})
axis([10^(0.8) 10^(4) 10^(-7) 10^(-0)]);
myLegend(0.9,-4,-7,style,txt)
ylabel('error and estimators')
xlabel('number of elements')
text(10^(3.7),10^(-5.1),'-3/2')
text(10^(2.5),10^(-0.9),'-2/3')
print -r600 -depsc2 fig5_hypsing_dofs.eps

%*** plot versus computation time
h = figure(2);
clf;
pos = get(h,'Position');
pos(1) = 0;
pos(2) = 0;
pos(3) = 1.5*pos(3);
pos(4) = 1.5*pos(4);
set(h,'Position',pos)
loglog(time_unif,err_unif+osc_unif,style{1},art{:})
hold on
loglog(time_unif,eta_unif,style{2},art{:})
loglog(time_unif,mu_tilde_unif,style{3},art{:})
loglog(time_unif,osc_unif,style{4},art{:})

loglog(time_adap,err_adap+osc_adap,style{5},art{:})
hold on
loglog(time_adap,eta_adap,style{6},art{:})
loglog(time_adap,mu_tilde_adap,style{7},art{:})
loglog(time_adap,osc_adap,style{8},art{:})
axis([10^(-1.3) 10^(4) 10^(-6) 10^(-0.5)]);
xlabel('computational time');
print -r600 -depsc2 fig5_hypsing_time.eps

%*** plot versus memory consumption
h = figure(3);
clf;
pos = get(h,'Position');
pos(1) = 0;
pos(2) = 0;
pos(3) = 1.5*pos(3);
pos(4) = 1.5*pos(4);
set(h,'Position',pos)
loglog(memory_unif/1048576,err_unif+osc_unif,style{1},art{:})
hold on
loglog(memory_unif/1048576,eta_unif,style{2},art{:})
loglog(memory_unif/1048576,mu_tilde_unif,style{3},art{:})
loglog(memory_unif/1048576,osc_unif,style{4},art{:})

loglog(memory_adap/1048576,err_adap+osc_adap,style{5},art{:})
hold on
loglog(memory_adap/1048576,eta_adap,style{6},art{:})
loglog(memory_adap/1048576,mu_tilde_adap,style{7},art{:})
loglog(memory_adap/1048576,osc_adap,style{8},art{:})
%axis([10^(-3) 10^(2.1) 10^(-4.5) 10^(-0.5)]);
xlabel('memory [MB]');
print -r600 -depsc2 fig5_hypsing_mem.eps
