function figure4_pointerror(varargin)
%FIGURE4_POINTERROR   Creates Figure 4 from HILBERT paper.
%   FIGURE4_POINTERR([NMAX]) provides an experimental comparison between 
%   uniform and adaptive mesh-refinement for Symm's integral equation, where
%   the given Dirichlet data are descretized either by nodal interpolation 
%   or the L2-proection. We consider a rotated L-shaped boundary, where the 
%   solution has a generic singularity, whereas the given Dirichlet data are
%   smooth. We plot the point error for some given point inside the domain.
%
%   We consider Symm's integral equation 
%
%      V phi = (K+1/2) uD
%
%   on an L-shaped boundary Gamma, which is equivalent to the Laplace 
%   problem
% 
%      -Laplace(u) = 0 in Omega with u = uD on Gamma
%
%   We prescribe the exact solution in polar coordinates
%
%      u(r,theta) = r^{2/3} * cos(2theta/3)
%
%   Then, phi is the normal derivative of u on Gamma.
%
%   For a given initial partition T0 with 8 uniform elements, we compute 
%   discrete solutions for both, uniform and adaptive mesh-refinement, as 
%   long as the generated meshes T have less than NMAX elements. If NMAX is
%   not specified as input of the function, we choose NMAX = 1000. We plot 
%   the weighted-residual error estimator, the data oscillations for the 
%   approximation of uD by uDh for nodal projection and L2-projection, as 
%   well as the nodal error  
%
%      | u(z) - uh(z) |  at the point z = [0.2, 0.13] in Omega
%
%   The approximate value is computed by means of the third Green's identity
%
%      uh(z) = V*phih(z) - K*uDh(z) 
%
%   For smooth exact solution, it is known that the nodal error in Omega 
%   decays with order h^2 for nodal interpolation and h^3 for uDh obtained
%   by L2-projection.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

if nargin > 0
    NMAX = varargin{1};
else
    NMAX = 1000;
end

%*** add hilberttools directory
addpath('../');

%*** load L-shaped domain
addpath('fig3to5/');
coordinates0 = load('coordinates.dat');
elements0 = load('elements.dat');

%*** rotate domain Omega so that exact solution is symmetric
alpha = 3*pi/4;
coordinates0 = coordinates0*[cos(alpha) -sin(alpha);sin(alpha) cos(alpha)]';

%*** shrink domain to ensure ellipticity of V by diam(Omega)<1
coordinates0 = coordinates0/4;

%*** evaluation point for Green's representation formula
z = [0.2 0.13];

%*** uniform
elements = elements0;
coordinates = coordinates0;
counter=1;
while 1

    fprintf('uniform --- number of elements: N = %d\r',size(elements,1))

    %*** compute data oscillations for nodal interpolation
    [osc_nodal,uDh_nodal] = computeOscDirichlet(coordinates,elements,@g);

    %*** compute data oscillations for L2 projection
    [osc_L2,uDh_L2] = computeOscDirichletL2(coordinates,elements,@g);
    
    %*** compute discrete solution and error estimator for nodal interpolation
    V = buildV(coordinates,elements);
    b_nodal = buildSymmRHS(coordinates,elements,uDh_nodal);
    x_nodal = V\b_nodal;
    rho_nodal = computeEstSlpResidual(coordinates,elements,x_nodal,uDh_nodal);
    
    %*** compute discrete solution for L2 projection
    b_L2 = buildSymmRHS(coordinates,elements,uDh_L2);
    x_L2 = V\b_L2;
    rho_L2 = computeEstSlpResidual(coordinates,elements,x_L2,uDh_L2);
    
    %*** compute nodal errors
    errz_nodal = abs( g(z) - evaluateV(coordinates,elements,x_nodal,z) ...
                           + evaluateK(coordinates,elements,uDh_nodal,z) );
    errz_L2 = abs( g(z) - evaluateV(coordinates,elements,x_L2,z) ...
                        + evaluateK(coordinates,elements,uDh_L2,z) );
                 
    %*** store computed data
    unif_dof(counter) = size(elements,1);
    unif_osc_nodal(counter) = sqrt(sum(osc_nodal));
    unif_osc_L2(counter ) = sqrt(sum(osc_L2));
    unif_rho_nodal(counter) = sqrt(sum(rho_nodal));
    unif_rho_L2(counter) = sqrt(sum(rho_L2));
    unif_errz_nodal(counter) = errz_nodal;
    unif_errz_L2(counter) = errz_L2;
        
    %*** increase counter, refine mesh and interate
    if size(elements,1) > NMAX
        break
    end
    counter = counter + 1;
    [coordinates,elements] = refineBoundaryMesh(coordinates,elements);  
end

%*** adaptive with nodal interpolation
theta = 0.25;
rho = 0.25;
coordinates = coordinates0;
elements = elements0;
counter=1;
while 1

    fprintf('adaptive (nodal interpolation) --- number of elements: N = %d\r', ...
            size(elements,1))
        
    %*** compute data oscillations for nodal interpolation
    [osc_nodal,uDh_nodal] = computeOscDirichlet(coordinates,elements,@g);

    %*** compute discrete solution and error estimator for nodal interpolation
    V = buildV(coordinates,elements);
    b_nodal = buildSymmRHS(coordinates,elements,uDh_nodal);
    x_nodal = V\b_nodal;
    rho_nodal = computeEstSlpResidual(coordinates,elements,x_nodal,uDh_nodal);
    
    %*** compute nodal error
    errz_nodal = abs( g(z) - evaluateV(coordinates,elements,x_nodal,z) ...
                           + evaluateK(coordinates,elements,uDh_nodal,z) );
   
    %*** store computed data
    nodal_dof(counter) = size(elements,1);
    nodal_osc(counter) = sqrt(sum(osc_nodal));
    nodal_rho(counter) = sqrt(sum(rho_nodal));
    nodal_errz(counter) = errz_nodal;
    
   
    %*** mark elements for refinement
    marked = markElements(theta,rho,rho_nodal + osc_nodal);
    
    %*** increase counter, refine mesh and interate
    if size(elements,1) > NMAX
      break;
    end
    counter = counter + 1;
    [coordinates,elements] = refineBoundaryMesh(coordinates,elements,marked);
    
end

%*** adaptive with L2-projection
theta = 0.25;
coordinates = coordinates0;
elements = elements0;
counter=1;
while 1

    fprintf('adaptive (L2-projection) --- number of elements: N = %d\r', ...
            size(elements,1))
        
    %*** compute data oscillations for nodal interpolation
    [osc_L2,uDh_L2] = computeOscDirichletL2(coordinates,elements,@g);

    %*** compute discrete solution and error estimator for nodal interpolation
    V = buildV(coordinates,elements);
    b_L2 = buildSymmRHS(coordinates,elements,uDh_L2);
    x_L2 = V\b_L2;
    rho_L2 = computeEstSlpResidual(coordinates,elements,x_L2,uDh_L2);
    
    %*** compute nodal error
    errz_L2 = abs( g(z) - evaluateV(coordinates,elements,x_L2,z) ...
                           + evaluateK(coordinates,elements,uDh_L2,z) );
   
    %*** store computed data
    l2_dof(counter) = size(elements,1);
    l2_osc(counter) = sqrt(sum(osc_L2));
    l2_rho(counter) = sqrt(sum(rho_L2));
    l2_errz(counter) = errz_L2;
     
    %*** mark elements for refinement
    marked = markElements(theta,rho,rho_L2 + osc_L2);
    
    %*** increase counter, refine mesh and interate
    if size(elements,1) > NMAX
      break;
    end
    counter = counter + 1;
    [coordinates,elements] = refineBoundaryMesh(coordinates,elements,marked);
    
end

%*** create plots
art = {'LineWidth',2,'MarkerSize',7};

style = {'r+-','rd-','bx--','bo--','md-','m<-','cx--','co--'};
txt = {'est (nodal, unif)','err (nodal, unif)', ...
    'est (l2, unif)','err (l2, unif)', ...
    'est (nodal, adap)','err (nodal, adap)', ...
    'est (l2, adap)','err (l2, adap)'};

loglog(unif_dof,unif_rho_nodal,style{1},art{:}); 
hold on;
loglog(unif_dof,unif_errz_nodal,style{2},art{:});
loglog(unif_dof,unif_rho_L2,style{3},art{:});
loglog(unif_dof,unif_errz_L2,style{4},art{:});
loglog(nodal_dof,nodal_rho,style{5},art{:});
loglog(nodal_dof,nodal_errz,style{6},art{:});
loglog(l2_dof,l2_rho,style{7},art{:});

loglog(l2_dof,l2_errz,style{8},art{:});

dof = [min([unif_dof,nodal_dof,l2_dof]),2*max([unif_dof,nodal_dof,l2_dof])];
loglog(dof,dof.^(-2/3),'k:','LineWidth',2);
loglog(dof,0.04*dof.^(-4/3),'k:','LineWidth',2);
loglog(dof,dof.^(-3/2),'k:','LineWidth',2);
loglog(dof,dof.^(-2),'k:','LineWidth',2);
loglog(dof,dof.^(-3),'k:','LineWidth',2);

text(10^3.5,1e-2,'O(2/3)')
text(10^4,2e-6,'O(3/2)')
text(2e4,1e-7,'O(4/3)')
text(4e3,3e-13,'O(3)')
text(1e4,1e-9,'O(2)')

myLegend(0.2,-7,-14,style,txt);
ylabel('error and error estimators')
xlabel('number of elements')

print -r600 -depsc2 fig4_pointerror.eps
