function figure3_symm(varargin)
%FIGURE3_SYMM   Creates Figure 3 from HILBERT paper.
%   FIGURE3_SYMM([NMAX]) provides an experimental comparison between uniform
%   and adaptive mesh-refinement for Symm's integral equation on a rotated
%   L-shaped boundary, where the solution as well as the Dirichlet data have
%   singularities of different order. Estimated error and error estimators
%   are plotted versus the number of degrees of freedom, the memory
%   consumption, and the CPU time.
%               
%   We consider Symm's integral equation 
%
%      V phi = (K+1/2) uD
%
%   on an L-shaped boundary Gamma, which is equivalent to the Laplace 
%   problem
% 
%      -Laplace(u) = 0 in Omega with u = uD on Gamma
%
%   For given delta>0, we define a function in polar coordinates
%
%      v_delta(r,theta) = r^delta * cos(delta*theta)
%
%   We prescribe the exact solution of the Laplace equation by
%
%      u(x) = v_{2/3}(x) + v_{7/8}(x-z)
%
%   where z is the uppermost corner of Gamma. Then, v has a generic 
%   singularity at the reentrant corner and an additional singularity at the
%   uppermost corner.
%
%   Then, the normal derivative phi of u has a generic singularity at the 
%   reentrant corner located at 0. Moreover, the given Dirichlet data have 
%   an additional singularity at the uppermost corner z.
%
%   For a given initial partition E0 with 8 uniform elements, we compute 
%   discrete solutions for both, uniform and adaptive mesh-refinement, as 
%   long as the generated meshes E have less than NMAX elements. If NMAX is
%   not specified as input of the function, we choose NMAX = 1000. We plot 
%   the (h-h/2)-error estimators eta, eta-tilde, mu, and mu-tilde as well as
%   a reliable upper bound for the error. All quantities are plotted over 
%   the number of elements, the memory consumption, and over the 
%   computational time.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

if nargin > 0
    NMAX = varargin{1};
else
    NMAX = 1000;
end

%*** add hilberttools directory
addpath('../');

%*** load L-shaped domain
addpath('fig3to5/');
coordinates0 = load('coordinates.dat');
elements0 = load('elements.dat');

%*** rotate domain Omega so that exact solution is symmetric
alpha = 3*pi/4;
coordinates0 = coordinates0*[cos(alpha) -sin(alpha);sin(alpha) cos(alpha)]';

%*** shrink domain to ensure ellipticity of V by diam(Omega)<1
coordinates0 = coordinates0/4;

%*** uniform
elements = elements0;
counter=1;
while 1
    
    %*** start time-measurement for uniform mesh-refinement
    time_ = cputime;

    %*** build mesh by successive uniform refinements of initial mesh
    coordinates = coordinates0;
    elements = elements0;
    for j = 1:counter-1
        [coordinates,elements] = refineBoundaryMesh(coordinates,elements);
    end
    
    fprintf('uniform --- number of elements: N = %d\r',size(elements,1))

    %*** compute coarse mesh solution
    V = buildV(coordinates,elements);
    [~,gh] = computeOscDirichlet(coordinates,elements,@g);
    b = buildSymmRHS(coordinates,elements,gh);
    x = V\b;

    %*** stop time-measurement for uniform mesh-refinement
    time_ = cputime - time_;

    %*** memory consumption
    s = whos('coordinates','elements','V','b','x');
    memory_ = 0;
    for k=1:size(s,1)
    	memory_ = memory_ + s(k).bytes;
    end

    %*** free obsolete memory
    clear V b osc gh s

    %*** compute fine mesh solution to estimate error
    [coordinates_fine,elements_fine,father2son] ...
      = refineBoundaryMesh(coordinates,elements);
    V_fine = buildV(coordinates_fine,elements_fine);
    [~,gh_fine] = computeOscDirichlet(coordinates_fine,elements_fine,@g);
    b_fine = buildSymmRHS(coordinates_fine,elements_fine,gh_fine);
    x_fine = V_fine\b_fine;
 
    %*** compute error estimators and reliable error bound
    eta_ = computeEstSlpEta(father2son,V_fine,x_fine,x);
    eta_tilde_ = computeEstSlpEtaTilde(father2son,V_fine,x_fine);
    mu_ = computeEstSlpMu(coordinates,elements,father2son,x_fine,x);
    mu_tilde_ = computeEstSlpMuTilde(coordinates,elements,father2son,x_fine);
    err_ = computeErrNeumann(coordinates,elements,x,@phi);
    osc_ = computeOscDirichlet(coordinates,elements,@g);

    %*** free obsolete memory
    clear V_fine b_fine x_fine osc_fine gh_fine x

    %*** store computed data
    N_unif(counter) = size(elements,1);
    time_unif(counter) = time_;
    memory_unif(counter) = memory_;
    eta_unif(counter) = sqrt(eta_);
    eta_tilde_unif(counter) = sqrt(eta_tilde_);
    mu_unif(counter) = sqrt(sum(mu_));
    mu_tilde_unif(counter) = sqrt(sum(mu_tilde_));
    err_unif(counter) = sqrt(sum(err_));
    osc_unif(counter) = sqrt(sum(osc_));

    %*** increase counter and iterate
    counter = counter+1;
    if (max(N_unif) > NMAX)
    	break
    end
end

%*** adaptive
theta = 0.25;
coordinates = coordinates0;
elements = elements0;
counter=1;
while 1

    fprintf('adaptive --- number of elements: N = %d\r',size(elements,1))
    
    %*** compute coarse-mesh solution
    V = buildV(coordinates,elements);
    [~,gh] = computeOscDirichlet(coordinates,elements,@g);
    b = buildSymmRHS(coordinates,elements,gh);
    x = V\b;

    clear V b osc gh

    %*** start time-measurement for adaptive mesh-refinement
    time_ = cputime;

    %*** compute fine-mesh solution
    [coordinates_fine,elements_fine,father2son] ...
        = refineBoundaryMesh(coordinates,elements);
    V_fine = buildV(coordinates_fine,elements_fine);
    [~,gh_fine] = computeOscDirichlet(coordinates_fine, ...
        elements_fine,@g);
    b_fine = buildSymmRHS(coordinates_fine,elements_fine,gh_fine);
    x_fine = V_fine\b_fine;

    %*** compute estimator mu_tilde and oscillations
    mu_tilde_ = computeEstSlpMuTilde(coordinates,elements, ...
        father2son,x_fine);
    osc_fine = computeOscDirichlet(coordinates_fine,elements_fine,@g);
    osc_ = osc_fine(father2son(:,1))+osc_fine(father2son(:,2));

    %*** mark elements for refinement and refine
    marked_elements ...
        = markElements(theta,mu_tilde_+osc_);

    [coordinates_new,elements_new] ...
        = refineBoundaryMesh(coordinates,elements,marked_elements);

    %*** stop time-measurement for adaptive mesh-refinement
    time_ = cputime - time_;
    if counter>1
        time_ = time_adap(counter-1)+time_;
    end

    %*** memory consumption
    s = whos('coordinates_fine','elements_fine', ...
        'coordinates_new','elements_new', ...
	'coordinates','elements', 'marked_elements', ...
	'V_fine','b_fine','N_fine','mu_tilde','osc_fine','osc_','x_fine');
    memory_ = 0;
    for k=1:size(s,1)
    	memory_ = memory_ + s(k).bytes;
    end

    %*** compute other estimators and error bound
    eta_ = computeEstSlpEta(father2son,V_fine,x_fine,x);
    eta_tilde_ = computeEstSlpEtaTilde(father2son,V_fine,x_fine);
    mu_ = computeEstSlpMu(coordinates,elements,father2son,x_fine,x);
    err_ = computeErrNeumann(coordinates,elements,x,@phi);  
        
    %*** store computed data
    N_adap(counter) = size(elements,1);
    time_adap(counter) = time_;
    memory_adap(counter) = memory_;
    eta_adap(counter) = sqrt(eta_);
    eta_tilde_adap(counter) = sqrt(eta_tilde_);
    mu_adap(counter) = sqrt(sum(mu_));
    mu_tilde_adap(counter) = sqrt(sum(mu_tilde_));
    osc_adap(counter) = sqrt(sum(osc_));
    err_adap(counter) = sqrt(sum(err_));

    %*** free memory
    clear V_fine x_fine b_fine father2son coordinates_fine elements_fine ...
        elements coordinates x osc gh osc_fine gh_fine

    %*** increase counter and iterate
    counter = counter+1;
    coordinates = coordinates_new;
    elements = elements_new;
    if max(N_adap) > NMAX
    	break
    end    
end

%*** create plots
art = {'LineWidth',2,'MarkerSize',7};
style = {'r+-.','gs-.','bo-.','c^-.', ...
         'r+-','gs-','bo-','c^-'};
txt = {'error (unif)','eta (unif)','tilde-mu (unif)','osc (unif)', ...
       'error (adap)','eta (adap)','tilde-mu (adap)','osc (adap)'};

%*** adaptive: plot only every second step for better overview
stop = size(N_adap,2);
N_adap = N_adap(1:2:stop);
time_adap = time_adap(1:2:stop);
memory_adap = memory_adap(1:2:stop);
eta_adap = eta_adap(1:2:stop);
err_adap = err_adap(1:2:stop);
mu_tilde_adap = mu_tilde_adap(1:2:stop);
osc_adap = osc_adap(1:2:stop);

%*** plot versus number of elements
h = figure(1);
clf;
pos = get(h,'Position');
pos(1) = 0;
pos(2) = 0;
pos(3) = 1.5*pos(3);
pos(4) = 1.5*pos(4);
set(h,'Position',pos)

loglog(N_unif,err_unif+osc_unif,style{1},art{:})
hold on
loglog(N_unif,eta_unif,style{2},art{:})
loglog(N_unif,mu_tilde_unif,style{3},art{:})
loglog(N_unif,osc_unif,style{4},art{:})
loglog(N_unif,N_unif.^(-2/3),'k--',art{:})

loglog(N_adap,err_adap+osc_adap,style{5},art{:})
hold on
loglog(N_adap,eta_adap,style{6},art{:})
loglog(N_adap,mu_tilde_adap,style{7},art{:})
loglog(N_adap,osc_adap,style{8},art{:})
loglog(N_unif,0.5*N_unif.^(-3/2),'k--',art{:})
axis([10^(0.8) 10^(4) 10^(-6) 10^(-0)]);
myLegend(0.9,-3.5,-6,style,txt)
ylabel('error and estimators')
xlabel('number of elements')
text(10^(2),10^(-3.8),'-3/2')
text(10^(2.5),10^(-1.5),'-2/3')
print -r600 -depsc2 fig3_symm_dofs.eps

%*** plot versus computation time
h = figure(2);
clf;
pos = get(h,'Position');
pos(1) = 0;
pos(2) = 0;
pos(3) = 1.5*pos(3);
pos(4) = 1.5*pos(4);
set(h,'Position',pos)
loglog(time_unif,err_unif+osc_unif,style{1},art{:})
hold on
loglog(time_unif,eta_unif,style{2},art{:})
loglog(time_unif,mu_tilde_unif,style{3},art{:})
loglog(time_unif,osc_unif,style{4},art{:})

loglog(time_adap,err_adap+osc_adap,style{5},art{:})
hold on
loglog(time_adap,eta_adap,style{6},art{:})
loglog(time_adap,mu_tilde_adap,style{7},art{:})
loglog(time_adap,osc_adap,style{8},art{:})
axis([10^(-1.3) 10^(4) 10^(-6) 10^(-0.5)]);
xlabel('computational time');
print -r600 -depsc2 fig3_symm_time.eps

%*** plot versus memory consumption
h = figure(3);
clf;
pos = get(h,'Position');
pos(1) = 0;
pos(2) = 0;
pos(3) = 1.5*pos(3);
pos(4) = 1.5*pos(4);
set(h,'Position',pos)
loglog(memory_unif/1048576,err_unif+osc_unif,style{1},art{:})
hold on
loglog(memory_unif/1048576,eta_unif,style{2},art{:})
loglog(memory_unif/1048576,mu_tilde_unif,style{3},art{:})
loglog(memory_unif/1048576,osc_unif,style{4},art{:})

loglog(memory_adap/1048576,err_adap+osc_adap,style{5},art{:})
hold on
loglog(memory_adap/1048576,eta_adap,style{6},art{:})
loglog(memory_adap/1048576,mu_tilde_adap,style{7},art{:})
loglog(memory_adap/1048576,osc_adap,style{8},art{:})
xlabel('memory [MB]');
print -r600 -depsc2 fig3_symm_mem.eps
