function figure2_threading(varargin)
%FIGURE2_THREADING   Creates Figure 2 from HILBERT paper.
%   FIGURE2_THREADING([THREADS]) provides an experimental comparison for the
%   use of multi-threading for the matrix assembly. The time for the 
%   assembly of V, K, W, and N is plotted over the number of threads.
%
%   We consider the L-shaped domain which is triangulated into 24.576 
%   triangles. The boundary is partitioned into 4.096 elements. For this 
%   discretization, we build the integral operator matrices V, K, W, and N
%   and measure the computational time in dependence of the threads used.
%   
%   The optional input parameter THREADS is an integer vector which contains
%   the number of threads which should be used for the multi-threaded 
%   assembly of the operators. If no array is provided, we use 
%   THREADS = [1, 2, 3, 4, 5, 6]. 
%   For each k = 1,...,length(THREADS) the function
%
%      o builds the MEX-functions for the operators by use of make
%      o assembles all integral operators and measures the time
%           
%   Afterwards, the assembly times are plotted over k.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

%*** change to hilberttools directory
cd ..

%*** add make directory
addpath('source/make/')

if nargin > 0
    threads = varargin{1};
else
    threads = [1, 2, 3, 4 , 5 , 6];
end

%*** load L-shaped domain
addpath('paper/fig2and6/')
vertices = load( 'coordinates.dat' );
triangles = load( 'triangles.dat' );

%*** refine the mesh uniform
[vertices, triangles, coordinates, elements] = ...
	buildSortedMesh( vertices, triangles );
for i=1:6
	[vertices, triangles, elements] = ...
		refineMesh( vertices, triangles, elements );
end
[vertices, triangles, coordinates, elements] = ...
	buildSortedMesh( vertices, triangles );

%*** refine the boundary mesh
elements_fine = elements;
vertices_fine = vertices;
triangles_fine = triangles;
for i=1:3
	[vertices_fine, triangles_fine, elements_fine] = ...
		refineMesh( vertices_fine, triangles_fine, elements_fine );
end
[vertices_fine, triangles_fine, coordinates_fine, elements_fine] = ...
	buildSortedMesh( vertices_fine, triangles_fine );

%*** build K, V, W, and N by use of multi-threading

generateConfigure( defaultConfigure() );
c = Configure();
data = zeros( 4, length(threads) );

for i = 1:length(threads)
    t = threads(i);
    make clean;
    if t == 1
	c.Threading = 'no';
    else
       	c.Threading = 'yes';
    end
    c.NumberOfCores = num2str(t);
    c.CompFlags = '-O2 -fPIC';
    generateConfigure( c );
    clear Configure;

    make;

    clear buildK buildN buildV buildW

    disp('')
    disp(['Matrix assembly uses ',int2str(t),' threads'])
    disp(['  Number of elements (for V, W, K): ' num2str(size(elements_fine, 1))])
    disp(['  Number of triangles (for N): ' num2str(size(triangles, 1))])

    disp(['  Build K']);
    tic;
    K = buildK( coordinates_fine, elements_fine );
    data( 1, i ) = toc;
    clear K;

    [vertices, triangles, coordinates_coarse, elements_coarse] = ...
		buildSortedMesh( vertices, triangles );
    disp(['  Build N']);
    tic;
    N = buildN( coordinates_coarse, elements_coarse, vertices, triangles );
    data( 2, i ) = toc;
    clear N;

    disp(['  Build V']);
    tic;
    V = buildV( coordinates_fine, elements_fine );
    data( 3, i ) = toc;
    clear V;

    disp(['  Build W']);
    tic;
    W = buildW( coordinates_fine, elements_fine );
    data( 4, i ) = toc;
    clear W;
end

%*** Visualize results:
art = {'LineWidth',2,'MarkerSize',7};
style = {'c+-','r^-','gs-','bo-'};
txt = {'buildK','buildN','buildV','buildW'};

figure(1)
clf
loglog(threads, data(3, :), style{1}, art{:}); %V
hold on
loglog(threads, data(1, :), style{2}, art{:}); %K
loglog(threads, data(4, :), style{3}, art{:}); %W
loglog(threads, data(2, :), style{4}, art{:}); %N
loglog(threads, 100*threads.^(-1), 'k--', art{:});
hold off
legend('buildVspace','buildK','buildW','buildN', '1/k' );
legend boxoff
xlabel( 'number of threads' );
ylabel( 'computational time' );

%*** change to paper/ directory
cd paper

print -r600 -depsc2 fig2_threading.eps
