function phix = phi(x,a,b)
%PHI   Neumann data for some Laplace problem.
%   PHIX = PHI(X,A,B) provides the Neumann data for some Laplace problem on
%   an L-shaped domain.
%
%   We treat the Laplace problem
% 
%      -Laplace(u) = 0 in Omega with u = g on Gamma
%
%   For given delta>0, we define a function in polar coordinates
%
%      v_delta(r,theta) = r^delta * cos(delta*theta)
%
%   We prescribe the exact solution of the Laplace equation by
%
%      u(x) = v_{alpha}(x) + v_{beta}(x-z)
%
%   where z is the uppermost corner of Gamma. Then, v has a generic 
%   singularity at the reentrant corner and an additional singularity at the
%   uppermost corner. Note that the function v_delta can be written in the 
%   complex plane by
%
%      v_delta(a+bi) = Real( (a+bi)^delta )
%
%   which is used for the implementation.
%
%   The evaluation points are given by the (N x 2)-matrix X. The 
%   (N x 2)-matrices A and B give the nodes of the corresponding boundary 
%   elements, i.e., X(j,:) belongs to the line segment 
%   Ej = conv{ A(j,:) , B(j,:) }. This information is used to compute the 
%   outer unit normal vector on Ej. The (N x 1)-vector Y provides the 
%   corresponding function values, i.e., PHIX(j) = phi(X(j,:)). 

% (C) 2009-2012 HILBERT-Team '09, '10, '12
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

%*** obtain parameters
[alpha,beta,z] = parameters();

%*** compute normal vector n on [a,b] for x in [a,b]
n = b-a;
n = [n(:,2) -n(:,1)] ./ repmat(sqrt(sum(n.^2,2)),1,2);

%*** compute gradient of v_alpha(x)
[t,r] = cart2pol(x(:,1),x(:,2));
grad1(:,1) = alpha*r.^(alpha-1) .* (cos(t).*cos(alpha*t)+sin(t).*sin(alpha*t));
grad1(:,2) = alpha*r.^(alpha-1) .* (sin(t).*cos(alpha*t)-cos(t).*sin(alpha*t));

%*** compute gradient of v_beta(x-z)
x = x - ones(size(x,1),1)*z;
[t,r] = cart2pol(x(:,1),x(:,2));
grad2(:,1) = beta*r.^(beta-1) .* ( cos(t).*cos(beta*t)+sin(t).*sin(beta*t) );
grad2(:,2) = beta*r.^(beta-1) .* ( sin(t).*cos(beta*t)-cos(t).*sin(beta*t) );

%*** compute normal derivative d_n u(x) = gradu(x)*n(x)
phix = sum((grad1 + grad2).*n,2);

