function gx = g(x)
%G   Dirichlet data for Laplace problem.
%   GX = G(X) provides the Dirichlet data for some homogeneous Laplace 
%   problem realized on an L-shaped domain.
%
%   We treat the Laplace problem
% 
%      -Laplace(u) = 0 in Omega with u = g on Gamma
%
%   For given delta>0, we define a function in polar coordinates
%
%      v_delta(r,theta) = r^delta * cos(delta*theta)
%
%   We prescribe the exact solution of the Laplace equation by
%
%      u(x) = v_{2/3}(x) + v_{7/8}(x-z)
%
%   where z is the uppermost corner of Gamma. Then, v has a generic
%   singularity at the reentrant corner and an additional singularity at the
%   uppermost corner. Note that the function v_delta can be written in the 
%   complex plane by
%
%      v_delta(a+bi) = Real( (a+bi)^delta )
%
%   which is used for the implementation.
%
%   The evaluation points are given by the (N x 2)-matrix X, and the
%   (N x 1)-vector Y provides the corresponding function values, i.e., 
%   GX(j) = g(X(j,:)).

% (C) 2009-2012 HILBERT-Team '09, '10, '12
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

%*** obtain parameters
[alpha,beta,z] = parameters();
z = z(1) + i*z(2);

%*** evaluate Dirichlet data
gx = real((x(:,1)+1i*x(:,2)).^alpha) + real( ((x(:,1)+1i*x(:,2)) - z ).^beta);

