function phix = phi(x,a,b)
%PHI   Neumann data for some Poisson problem.
%   PHIX = PHI(X,A,B) provides the Neumann data for some Poisson problem on
%   an L-shaped domain.
%
%   We treat the Poisson problem
% 
%      -Laplace(u) = f in Omega with du/dn = phi on Gamma (or GammaN)
%
%   We prescribe the exact solution by
%
%      u(x) = |x-z|^(2\beta) + r^{2/3} * cos(2theta/3)
%
%   where alpha = 2/3, beta = 0.9, and z = [0.14 0.14].
%
%   The evaluation points are given by the (N x 2)-matrix X. The 
%   (N x 2)-matrices A and B give the nodes of the corresponding boundary 
%   elements, i.e., X(j,:) belongs to the line segment 
%   Ej = conv{ A(j,:) , B(j,:) }. This information is used to compute the 
%   outer unit normal vector on Ej. The (N x 1)-vector Y provides the 
%   corresponding function values, i.e., PHIX(j) = phi(X(j,:)). 

% (C) 2010-2012 HILBERT-Team '10, '12
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

%*** exact solution is u(x) = u1(x) + u2(x), where
%***   u1(x) = Real( x1+i*x2)^alpha );
%***   u2(x) = |x-z|^(2*beta) = [ (x1-z1)^2 + (x2-z2)^2 ]^beta

%*** obtain parameters
[alpha,beta,z] = parameters();

%*** compute gradient of u1
[t,r]=cart2pol(x(:,1),x(:,2));

d1u1 = alpha*r.^(alpha-1) .* ( cos(t).*cos(alpha*t)+sin(t).*sin(alpha*t) );
d2u1 =alpha*r.^(alpha-1) .* ( sin(t).*cos(alpha*t)-cos(t).*sin(alpha*t) );

%*** compute gradient of u2
x = x - repmat(z,length(x),1);
d1u2 = 2*beta*x(:,1).*sum(x.^2,2).^(beta-1);
d2u2 = 2*beta*x(:,2).*sum(x.^2,2).^(beta-1);

%*** compute normal vector n on [a,b] for x in [a,b]
n = b-a;
n = [n(:,2) -n(:,1)] ./ repmat(sqrt(sum(n.^2,2)),1,2);

%*** evaluate normal derivative phi
gradx = [d1u1 + d1u2 , d2u1 + d2u2];
phix = sum(gradx.*n,2);
