function make(varargin)
%MAKE   Builds HILBERT's operators.
%
%    Available targets and their description:
%       help:            shows this list of available targets.
%       all:             builds all MEX functions that belong to HILBERT.
%       clean:           deletes all files that are created during the 
%                        building of HILBERT's MEX functions.
%       realclean:       like clean, but also deletes the configuration 
%                        file.
%       configure:       interactivly creates a configuration file.
%       configure-gui:   runs the graphical dialogs to interactivly create a
%                        configuration file.
%       configure-text:  runs the text-based configuration assistant to
%                        interactivly create a configuration file.
%
%   Furthermore there is a target named function_name for each of HILBERT's
%   MEX functions.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

    addpath source/make;

    if (nargin == 0)
        make all;
        return;
    elseif (nargin > 1)
        for argin=varargin
            make(argin);
        end
        return;
    end
    
    target_name = varargin{1};

    %*** Load an alternative Makefile, if a function handle is provided.
    if (strcmp(class(target_name), 'function_handle') == true)
      getTargetByName(target_name);
      return;
    end
    
    %*** This code recursivly builds the target named target_name.
    t = getTargetByName(target_name);
    if (isstruct(t) == false) % t is a file-target
        tf = fileAvailable(target_name);
        if (tf == false)
            error(['Target ' target_name ...
                                ' does not exist and is not a file.']);
        end
    elseif (t.available_handler(t) == false)
        for depend=t.dependencies
            dependstr = char(depend);
            depend_target = getTargetByName(dependstr);
            if (isstruct(depend_target) == false)
                if (fileAvailable(dependstr) ~= false)
                    continue;
                end
                error(['Target ' dependstr ...
                                ' does not exist and is not a file.']);
            end

            if (depend_target.available_handler(depend_target) == false)
                make(dependstr);
            end
        end

        try
            build_status = t.build_handler(t);
        catch
            disp('Uncaught build handler exception:');
            disp('This is probably a severe and unintended error.');
            disp('Please run "make realclean" to clean up');
            error('Uncaught build handler exception!');
        end

 
        if (build_status == false)
            error(['Failed when building ' t.name]);
        end
    else
        disp(['The target ' target_name ' is already up-to-date.']);
    end
    
    disp(['Built target ' target_name ' successfully.']);

    return;
end

function targets = Makefile()
%MAKEFILE   List of targets. 
%   MAKEFILE returns a list of targets that is processed by make.m and its
%   helpers.
%
%   This file is part of the building system for HILBERT. HILBERT's
%   building system is a recursive, dependency-based system similar to 
%   "make".
%
%   This file contains a list of targets and their dependencies between each
%   other. It also contains some functions that contain the code that is 
%   responsible for building certain targets.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

  addpath source/make/dialogs;
  
  createEmptyConfigure();
  
  targets = [ ...
    ... % Essential files:
    addObjectFiles('buildN', 'buildK', 'buildV', 'buildW', 'constants', ...
        'evaluateK', 'evaluateKadj', 'evaluateV', 'evaluateW', 'evaluateN', ...
        'doubleLayerPotential', 'gaussQuadrature', 'geometry', ...
        'newtonPotential', 'singleLayerPotential', ...
        'threadedK', 'threadedN', 'threadedV', 'threadedW', 'threads'), ...
    ... % Target for the operators:
    addThrMexTarget('buildN', {'buildN.o', ...
        'constants.o', 'gaussQuadrature.o', 'geometry.o', ...
        'singleLayerPotential.o', 'newtonPotential.o', 'threads.o', ...
        'threadedN.o'}), ...
    addThrMexTarget('buildK', {'buildK.o', 'doubleLayerPotential.o', ...
        'constants.o', 'gaussQuadrature.o', 'geometry.o', ...
        'threads.o', 'threadedK.o'}), ...
    addThrMexTarget('buildV', {'buildV.o', ...
        'constants.o', 'gaussQuadrature.o', 'geometry.o', ...
        'singleLayerPotential.o', 'threads.o', 'threadedV.o'}), ...
    addThrMexTarget('buildW', {'buildW.o', ...
        'constants.o', 'gaussQuadrature.o', 'geometry.o', ...
        'singleLayerPotential.o', 'threads.o', 'threadedW.o'}), ...
    addMexTarget('evaluateK', {'evaluateK.o', ...
        'constants.o', 'gaussQuadrature.o', 'geometry.o', ...
        'doubleLayerPotential.o'}), ...
    addMexTarget('evaluateKadj', {'evaluateKadj.o', ...
        'constants.o', 'gaussQuadrature.o', 'geometry.o', ...
        'doubleLayerPotential.o'}), ...
    addMexTarget('evaluateV', {'evaluateV.o', ...
        'constants.o', 'gaussQuadrature.o', 'geometry.o', ...
        'singleLayerPotential.o'}), ...
    addMexTarget('evaluateW', {'evaluateW.o', ...
        'constants.o', 'gaussQuadrature.o', 'geometry.o', ...
        'doubleLayerPotential.o'}), ...
    addMexTarget('evaluateN', {'evaluateN.o', ...
        'constants.o', 'gaussQuadrature.o', 'geometry.o', ...
        'newtonPotential.o', 'singleLayerPotential.o'}), ...
    ... % Meta targets:
    createTarget('all', {'buildN','buildK','buildV','buildW', ...
          'evaluateK', 'evaluateKadj', 'evaluateV', 'evaluateW', 'evaluateN'},...
        @installWindowsDlls, @neverAvailable), ...
    ... % Cleaning up
    createTarget('clean', {}, @clean, @neverAvailable), ...
    createTarget('clear', {'clean'}, @nullAction, @neverAvailable), ...
    createTarget('realclean', {'clean'}, @realclean, @neverAvailable), ...
    ... % Configuration
    createTarget('configure-gui',{},@configureGui,@neverAvailable),...
    createTarget('configure-text',{},@configureText,@neverAvailable),...
    createTarget('configure',{},@configure,@neverAvailable), ...
    createTarget('clean-build',{'clean', 'all'},@nullAction,@neverAvailable),...
    ... % Windows installation target
    createTarget('win-install',{},@installPrecompiled,@neverAvailable),...
    ... % Help:
    createTarget('help', {}, @help, @neverAvailable) ...
  ];
end

%*** Deletes the configuration file.
function status = realclean(target)
  status = true;

  try
    delete source/make/Configure.m
    createEmptyConfigure();
  catch
    status = false;
  end
end

%*** Deletes all object-files and MEX-files.
function status = clean(target)
  try
    clear all;
    status = true;
    delete 'source/*.o'
    delete 'source/*.obj'
    delete(['*.' mexext()]);
    delete '*.dll'
  catch
    status = false;
  end
end

%*** Displays the help message.
function status = help(target)
  disp(sprintf([ ...
  '*** List of available targets:\n' ...
  '\thelp:    shows this list of available targets.\n' ...
  '\tall:     compiles all MEX functions that belong to HILBERT.\n' ...
  '\tclean:   deletes all files that are created during the building of\n'... 
  '\t         the HILBERT MEX files.\n' ...
  '\trealclean: like clean, but also deletes the configuration file.\n' ...
  '\tconfigure: interactivly creates a Configure.m file.\n' ...
  '\tconfigure-gui: runs the graphical configuration dialog.\n' ...
  '\tconfigure-text: runs the text-based configuration dialog.\n' ...
  '\n' ...
  'Furthermore for each function in HILBERT there is a target\n' ...
  'function_name that builds this function.\n' ...
  ]));
  status = true;
end

function status = configure(target)
if feature('ShowFigureWindows')
  try
    status = configureGui(target);
  catch
    status = configureText(target);
  end
else
  disp('*********************************************************************')
  disp('Note: No windows can be displayed in your MATLAB session.')
  disp('      I am running a text-based version of make.')
  disp('*********************************************************************')
  disp(' ')  
  status = configureText(target);
end
  
end

function status = configureGui(target)
if feature('ShowFigureWindows')
  try
    config = DialogConfig();
    generateConfigure(config);
    clear Configure;
    tf = DialogMakeCleanBuild();
    if (tf)
      make clean-build;
    end
    status = 1;
  catch
    disp lasterr();
    status = 0;
  end
else
  disp('*********************************************************************')
  disp('Note: No windows can be displayed in your MATLAB session.')
  disp('      I am running a text-based version of make.')
  disp('*********************************************************************')
  disp(' ')
  status = configureText(target);
end


end

function status = installWindowsDlls(target)
  try
    if (ispc())
      copyfile(['source\make\ext\lib\' mexext '\pthreadVC2.dll'], ...
               'pthreadVC2.dll');
    end
    status = 1;
  catch
    disp lasterr();
    status = 0;
  end
end
  
function status = installPrecompiled(target)
  config = Configure();
  try
    installWindowsBinaries(config);
    status = 1;
  catch
    e = lasterror();
    disp(e.message);
    status = 0;
  end
end

function status = mexCompile(target)
  cflags = '';
  if (exist('Configure', 'file') == 2)
    c = Configure();
    cflags = [' -DEPS=' c.Eps ...
              ' -DDEFAULT_ETA=' c.Eta ...
              ' -DGAUSS_ORDER=' c.GaussOrder ];
    if (strcmp(c.Threading, 'yes'))
      cflags = [cflags ' -DHILTHREADS -DNUMOF_CORES=' c.NumberOfCores];
      if (strcmp(c.ThreadingDiagnostics, 'yes'))
        cflags = [cflags ' -DDIAGNOSTICS'];
      end
    end
    if (strcmp(c.Debugging, 'yes'))
      cflags = [cflags ' -g'];
    else
      cflags = [cflags ' -DNDEBUG'];
    end
    if (~strcmp(c.CompFlags, ''))
      if (ispc)
        cflags = [cflags ' COMPFLAGS="$COMPFLAGS ' c.CompFlags '"'];
      elseif (isunix)
        cflags = [cflags ' CFLAGS="$CFLAGS ' c.CompFlags '"'];
      else
        disp(['Warning: I do not know how to use compiler flags for your' ...
              ' system.']);
      end
    end
  end
  output_file = [target.name];
  if (ispc)
    output_file = [output_file 'bj'];
  end
  build_str = ['mex ' cflags ' -c' ...
                  ' -I.' ...
                  ' -outdir source/ -output ' output_file ' "' ...
                  char(target.dependencies(end)) '"'];
  if (ispc)
      build_str = [build_str ' -Isource\make\ext\include'];
  end
  disp(build_str);
  eval(build_str);
  status = objectFileAvailable(target);
end

function status = mexLinkWithThreading(target)
  c = Configure();
  build_str = 'mex ';
  if (strcmp(c.Debugging, 'yes'))
      build_str = [build_str '-g '];
  end
  if (isunix)
      build_str = [build_str ' -lpthread -cxx'];
  else
      libpath = ['source\make\ext\lib\' mexext '\'];
      build_str = [build_str '-L"' libpath '" -lpthreadVC2 '];
  end
  build_str = [build_str ' -output ./' target.name '.' mexext];
                 
  for d=target.dependencies
      build_str = [build_str ' source/' char(d)];
      if (ispc)
          build_str = [build_str 'bj'];
      end
  end
  
  disp(build_str);
  eval(build_str);
  
  if (ispc && strcmp(c.Threading, 'yes') && ...
          ~fileAvailable('pthreadVC2.dll'))
      disp('Installing threading run-time ...');
      copyfile(['source\make\ext\lib\' mexext '\pthreadVC2.dll'], ...
               'pthreadVC2.dll');
  end
  
  status = fileAvailable([target.name '.' mexext]);
end

function status = mexLink(target)
  c = Configure();
  build_str = 'mex ';
  if (strcmp(c.Debugging, 'yes'))
      build_str = [build_str '-g '];
  end
  if(isunix())
      build_str = [build_str '-cxx -output ' target.name '.' mexext];
  else
      build_str = [build_str '-output ' target.name '.' mexext];
  end
  
  for d=target.dependencies
      build_str = [build_str ' source/' char(d)];
      if (ispc)
          build_str = [build_str 'bj'];
      end
  end
  
  disp(build_str);
  eval(build_str);
  status = fileAvailable([target.name '.' mexext]);
end

function target = addMexTarget(name, dep)
  target = createTarget(name, dep, @mexLink, @mexBuildAvailable);
end

function target = addThrMexTarget(name, dep)
  target = createTarget(name, dep, @mexLinkWithThreading, @mexBuildAvailable);
end

function targets = addObjectFiles(varargin)
  SEARCH_PATH = { [ pwd '/source' ] };

  targets = [];

  for i=1:nargin
    for path = SEARCH_PATH
      src_file = [path{:} '/' varargin{i} '.c'];
      h_file = [path{:} '/' varargin{i} '.h'];
      if (exist(src_file, 'file') == 2)
        break;
      end
    end

    if (~exist(src_file, 'file') == 2)
      error(['Could not locate source file in ' SEARCH_PATH '!']);
    end

    if (exist(h_file, 'file') == 2)
      dep = {h_file, src_file};
    else
      dep = {src_file};
    end

    targets = [targets, ...
      createTarget([varargin{i} '.o'],dep,@mexCompile,@objectFileAvailable)];
  end
end

function availability = mexBuildAvailable(target)
file_name = [target.name '.' mexext()];
  c = Configure();
  if (strcmp(c.BinaryInstall, 'yes'))
    if (fileAvailable(file_name))
      availability = true;
    else
      availability = false;
    end
  else
    mtime = fileAvailable(file_name);
    availability = targetUpToDate(target, mtime);
  end
end

function availability = objectFileAvailable( target )
  file_name = ['source/' target.name];
  if (ispc)
    % In windows the extension for object files is .obj and not .o.
    file_name = [file_name 'bj'];
  end
  availability = fileAvailable(file_name);
end

function availability = configAvailable(target)
  availability = fileAvailable('source/make/Configure.m');
end

function createEmptyConfigure(target)
  if (~ configAvailable())
    copyfile('source/make/EmptyConfigure.m', 'source/make/Configure.m');
  end
end

function config = genConfig(varargin)
  try
    config = guiConfig(varargin);
  catch
    config = textConfig(varargin);
  end
end

function config = guiConfig(varargin)
  config = DialogConfig(varargin);
end

function target = createTarget(name, dependencies, build_handler, varargin)
%CREATETARGET   Is a helper function for building a target structure.
%   CREATETARGET(name, dependencies, build_handler [, available_handler])
%   builds a target structure.
%
%   This file is part of the building system for HILBERT. HILBERT's building
%   system is a recursive, dependency-based system similar to "make".
%
%   CREATETARGET is a helper function for building a target structure. Each
%   target is represented by a structure that has the fields "name", 
%   "dependencies", "build_handler" and "available_handler". "name" is the 
%   target's name, "dependencies" is a (cell) array of strings containing 
%   the targets' names, this target depends on. "build_handler" is a 
%   function handle for the function that contains the code for building the
%   target. "available_handler" is a function handle to the function that
%   checks whether that target is available or not. Both, the build and the
%   availability handler get a copy of the target structure when called.
%
%   The build handler must return true if it built the target successfully 
%   and false in case it did not. It may use the availability handler as an
%   indicator of success.
%
%   The availability handler must return either true, false or a number 
%   representing the last modification time of that target. true is 
%   returned, in case the target is available and it cannot become obsolete.
%   This might be useful, in case a target provides a certain resource, like
%   a network connection. The handler must return false, in case a target is
%   unavailable and the last modification time (in MATLAB's date/time 
%   format) in case the target is available and may become obsolete.
%
%   One may omit the availability handler when calling createTarget. In this
%   case, the default availability handler "isAvailable" is chosen.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

  if nargin == 4
      available_handler = varargin(1);
  else
      available_handler = @isAvailable;
  end

  target = struct('name', name, 'dependencies', ...
                  {dependencies} ,...
                  'build_handler', build_handler, ...
                  'available_handler', available_handler);
end

function availability = fileAvailable(file_name)
  if (exist(file_name, 'file') == 2 || exist(file_name, 'file') == 3)
    try
      info = dir(file_name);
      availability = info.datenum;
    catch
      availability = false;
    end
  else
    availability = false;
  end
end

function return_target = getTargetByName(name)
  global targets;

  if (strcmp(class(name), 'function_handle') == true)
    targets = name();
    if (all(size(targets) == [0 0]))
      error('Makefile has no targets.');
    end
    return;
  elseif (all(size(targets) == [0 0]))
    targets = Makefile();
  end

  if (isstr(name) == false)
    error(['Expecting either a string or a function handle as input ' ...
      'parameter, but got ' class(name) '.']);
  end

  for target=targets
    if (strcmp(target.name, name))
      return_target = target;
      return;
    end
  end
  return_target = false;
end

function availability = isAvailable(target)
  mtime = fileAvailable(target.name);
  availability = targetUpToDate(target, mtime);
end

function availability = neverAvailable(target)
  availability = false;
end

function status = nullAction(target)
  status = true;
end

function status = targetUpToDate(target, mod_time)
  status = false;

  for depname = target.dependencies
    dep = getTargetByName(depname{:});
    if (islogical(dep) && dep == false)
      % Target does not exist. Is it a file?
      dfmtime = fileAvailable(depname{:});
      if (islogical(dfmtime) && dfmtime == false)
        error(sprintf('%s is neither a file nor a target.\n', depname{:}));
      end
      if (dfmtime > mod_time)
        return;
      end
    else
      depstat = dep.available_handler(dep);
      if (islogical(depstat))
        if (depstat == false)
          return;
        end
      else
        if (depstat > mod_time)
          return;
        elseif (targetUpToDate(dep, mod_time) == false)
          dep.build_handler(dep);
          return;
        end
      end
    end
  end

  status = mod_time;
end

