function fx = f(x)
%F   Volume data for some Poisson problem.
%   FX = F(X) provides the volume data for some Poisson problem on an 
%   L-shaped domain.
%
%   We treat the Poisson problem
% 
%      -Laplace(u) = f in Omega
%
%   We prescribe the exact solution by
%
%      u(x) = |x-z|^(2\beta) + r^{2/3} * cos(2theta/3)
%
%   where alpha = 2/3, beta = 0.9, and z = [0.14 0.14].
%
%   The evaluation points are given by the (N x 2)-matrix X. The 
%   (N x 1)-vector FX provides the corresponding function values, i.e., 
%   FX(j) = F(X(j,:)). 

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

%*** exact solution is u(x) = u1(x) + u2(x), where
%***   u1(x) = Real( x1+i*x2)^alpha );
%***   u2(x) = |x-z|^(2*beta) = [ (x1-z1)^2 + (x2-z2)^2 ]^beta
[~,beta,z] = parameters();

%*** second-order derivatives d2u/dx1 and d2u/dx2
x = x - repmat(z,length(x),1);
d11u = 2*beta*sum(x.^2,2).^(beta-1) ...
     + 4*beta*(beta-1)*x(:,1).^2.*sum(x.^2,2).^(beta-2);
 
d22u = 2*beta*sum(x.^2,2).^(beta-1) ...
     + 4*beta*(beta-1)*x(:,2).^2.*sum(x.^2,2).^(beta-2);

%*** f = -Laplace(u)
fx  = -(d11u + d22u);
