%ADAPTIVESYMMVOLMU   Demo for Symms IE with volume force and mu estimator.
%   ADAPTIVESYMMVOLMU provides the implementation of an adaptive mesh-
%   refining algorithm for Symms integral equation with volume force,
%   steered by the (h-h/2)-type error estimator mu. We consider
%
%      V*phi = (K+1/2)*g -Nf
%
%   which is equivalent to the Dirichlet problem
% 
%      -Laplace(u) = f in Omega with u = g on Gamma
%
%   We use functionality provided by the HILBERT program package. As example
%   serves an L-shaped domain Omega with known exact solution
%
%      u(x) = |x-z|^(1.8) + r^(2/3)*cos(2 \theta / 3)
%
%   with z = [0.14 0.14], prescribed in polar coordinates.
%   The exact solution u exhibits a generic singularity. Given an initial
%   mesh, the adaptive algorithm reads as follows:
%
%   (1) build Galerkin data w.r.t. (Ecoarse,Tcoarse) and compute data
%       oscillations.
%   (2) compute Galerkin solution w.r.t (Ecoarse,Tcoarse)
%   (3) refine given boundary mesh Ecoarse uniformly and obtain
%       Efine
%   (4) build Galerkin data w.r.t. (Efine,Tcoarse)
%   (5) compute Galerkin solution w.r.t. (Efine,Tcoarse)
%   (6) compute Ecoarse-elementwise contributions of
%       mu and Dirichlet data oscillations,
%       compute Tcoarse-elementwise contributions of
%       volume data oscillations
%   (7) use Doerfler strategy to mark certain
%       (Ecoarse,Tcoarse)-elements
%   (8) refine marked elements to obtain new mesh
%       (Ecoarse,Tcoarse)
%   (9) iterate as long as #(Ecoarse,Tcoarse) is lower than
%       given bound
%
%   At the end, the error, the estimator as well as the oscillations in each
%   step of the algorithm are plotted over the number of boundary elements. 
%   Additionally, we plot the BEM solution vs. the exact solution over the
%   arclength.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

%*** add hilberttools main directory
addpath('../../'); 

%*** load L-shaped domain Omega
vertices = load('coordinates.dat');
triangles = load('triangles.dat');

%*** rotate domain Omega so that exact solution is symmetric
alpha = 3*pi/4;
vertices = vertices*[cos(alpha) -sin(alpha);sin(alpha) cos(alpha)]';

%*** shrink domain to ensure ellipticity of V by diam(Omega)<1
vertices = vertices/4;

%*** maximal number of elements
nEmax = 500;

%*** adaptivity parameter
theta = 0.25;

%*** counter
j = 1;

%*** store number of elements, exact error, error estimator, and
%*** data oscillations in each step of the adaptive loop
nE = zeros(2,1);
err = zeros(2,1);
est = zeros(2,1);
oscillations = zeros(2,1);
oscillationsVol = zeros(2,1);

%*** extract boundary mesh
[vertices,triangles,coordinates,elements] = ...
    buildSortedMesh(vertices,triangles);

%*** adaptive mesh-refining algorithm
while 1
    
    fprintf('number of elements: N = %d (Gamma), %d (Omega)\r', ...
        size(elements,1),size(triangles,1));

    %*** discretize data and compute data oscillations on coarse mesh
    [osc,gh] = computeOscDirichlet(coordinates,elements,@g);
    [oscV,fh] = computeOscVolume(vertices,triangles,@f);

    %*** compute coarse-mesh solution
    V = buildV(coordinates,elements);
    b = buildSymmVolRHS(coordinates,elements,gh,vertices,triangles,fh);
    x = V\b;

    %*** build uniformly refined mesh
    [coordinates_fine,elements_fine,father2son] ...
        = refineBoundaryMesh(coordinates,elements);

    %*** discretize Dirichlet data and compute data oscillations on coarse
    %*** mesh
    [~,gh_fine] = computeOscDirichlet(coordinates_fine,elements_fine,@g);
    
    %*** compute fine-mesh solution
    V_fine = buildV(coordinates_fine,elements_fine);
    b_fine = buildSymmVolRHS(coordinates_fine,elements_fine,gh_fine, ...
        vertices,triangles,fh);
    x_fine = V_fine\b_fine;

    %*** compute (h-h/2)-error estimator mu
    mu = computeEstSlpMu(coordinates,elements,father2son,x_fine,x);

    %*** the following lines are only used for visualization and are not
    %*** part of the adaptive algorithm
    nE(j) = size(elements,1);
    err(j) = sqrt(sum(computeErrNeumann(coordinates,elements,x,@phi)));
    est(j) = sqrt(sum(mu));
    oscillations(j) = sqrt(sum(osc));
    oscillationsVol(j) = sqrt(sum(oscV));

    %*** mark elements for refinement
    [marked_elements,marked_triangles] = markElements(theta,mu + osc,oscV);

    %*** generate new mesh
    [vertices,triangles,elements] ...
        = refineMesh(vertices,triangles,elements,marked_triangles,marked_elements);
    coordinates = vertices(unique(elements),:);

    %*** stopping criterion
    if (size(elements,1) + size(triangles,1)) > nEmax
      break;
    else
      j = j+1;
    end
end

%*** the user should read the help section of this demo
disp(' ');
disp('Please read the documentation of this demo for details on the ');
disp('adaptive loop and on the problem considered.');
disp(['To that end, type ''help ',mfilename,''' at the MATLAB prompt.']);

%*** visualize error, estimator, and oscillations
figure(1);
clf;
loglog(nE,err,'--x',nE,est,'--o',nE,oscillations,'-->',nE,oscillationsVol,'--<',nE,nE.^(-3/2),'--k');
xlabel('number of boundary elements N');
ylabel('error bound, estimator and oscillations');
legend('error','mu','osc','volume osc','O(N^{-3/2})');
axis tight

%*** visualize exact and adaptively computed solution
figure(2);
clf;
plotArclengthP0(coordinates_fine,elements_fine,x_fine,@phi);
title('BEM solution vs. exact solution');
xlabel('arclength from origin');
legend('BEM solution','exact solution');
