%ADAPTIVEMIXEDVOLRESIDUAL   Demo for the mixed problem with rho estimator.
%   ADAPTIVEMIXEDVOLRESIDUAL provides the implementation of an adaptive
%   mesh-refining algorithm for the mixed model problem on the L-shape,
%   steered by the residual-type error estimator rho. We consider
%
%      -K*gN + V*phiD = (1/2+K)*gD - V*phiN  -N0*f
%      W*gN + K'*phiD = -W*gD +(1/2-K')*phiN -N1*f
%
%   which is equivalent to the mixed boundary value problem
%
%      -Laplace(u)  = f    in Omega
%               u   = gD   on GammaD
%             du/dn = phiN on GammaN
%
%   We use functionality provided by the HILBERT program package. As example
%   serves an L-shaped domain Omega with known exact solution
%
%      u(x) = |x-z|^(1.8) + r^(2/3)*cos(2 \theta / 3)
%
%   with z = [0.14 0.14], prescribed in polar coordinates.
%   The exact solution u exhibits a generic singularity. Given an initial
%   mesh, the adaptive algorithm reads as follows:
%
%   (1) build Galerkin data and compute Dirichlet/Neumann/volume data
%       oscillations w.r.t. (E,T)
%   (2) compute Galerkin solution w.r.t. (E,T)
%   (3) compute E-elementwise contributions of eta on GammaD and
%       on GammaN
%   (4) use Doerfler strategy to mark certain (E,T)-elements
%   (5) refine marked elements to obtain new mesh T
%   (6) iterate as long as #T+#E is lower than given bound
%
%   At the end, the error, the estimator as well as the oscillations in each
%   step of the algorithm are plotted over the number of elements. 
%   Additionally, we plot the BEM solution vs. the exact solution over the
%   arclength.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

%*** add hilberttools main directory
addpath('../../'); 

%*** add hilberttools main directory
addpath('../../');

%*** load L-shaped domain Omega
vertices = load('coordinates.dat');
triangles = load('triangles.dat');

%*** split Gamma into Dirichlet and Neumann boundary
dirichlet = [1 2;2 3;3 6;6 5];
neumann = [5 8;8 7;7 4;4 1];

%*** rotate domain Omega so that exact solution is symmetric
alpha = 3*pi/4;
vertices = vertices*[cos(alpha) -sin(alpha);sin(alpha) cos(alpha)]';

%*** shrink domain to ensure ellipticity of V by diam(Omega)<1
vertices = vertices/4;

%*** extract boundary mesh
[vertices,triangles,coordinates,neumann,dirichlet] = ...
    buildSortedMesh(vertices,triangles,neumann,dirichlet);

%*** maximal number of elements
nEmax = 500;

%*** adaptivity parameter
theta = 0.25;

%*** counter
j = 1;

%*** store number of elements, exact error, error estimator, and data
%*** oscillations in each step of the adaptive loop
nE = zeros(2,1);
err = zeros(2,1);
est = zeros(2,1);
oscillations = zeros(2,1);
oscillationsVol = zeros(2,1);

%*** adaptive mesh-refining algorithm
while 1

    fprintf('number of elements: N = %d (Gamma) %d (Omega)\r',...
        size(neumann,1)+size(dirichlet,1),size(triangles,1));

    %*** rearrange indices such that Neumann nodes are first
    [vertices,triangles,coordinates,neumann,dirichlet] = ...
        buildSortedMesh(vertices,triangles,neumann,dirichlet);
    
    %*** discretize data and compute corresponding data oscillations for
    %*** fine mesh
    [oscD,oscN,gDh,phiNh] ...
        = computeOscMixed(coordinates,dirichlet,neumann,@g,@phi);
    [oscV,fh] = computeOscVolume(vertices,triangles,@f);	  
    
    %*** compute integral operators
    elements = [dirichlet;neumann];
    V = buildV(coordinates,elements);
    K = buildK(coordinates,elements);  
    W = buildW(coordinates,elements);
    
    %*** compute right-hand side     
    [b_nodes,b_elements,lambdah] =...
        buildMixedVolRHS(coordinates,dirichlet, ...
        neumann,gDh,phiNh,vertices,triangles,fh,V,K,W);

    %*** compute degrees of freedom
    nC = size(coordinates,1);
    nD = size(dirichlet,1);
    freeNeumann = setdiff(1:nC,unique(dirichlet));
    nN=length(freeNeumann);
    freeDirichlet = 1:nD;

    W = W(freeNeumann,freeNeumann);
    K = K(freeDirichlet,freeNeumann);
    V = V(freeDirichlet,freeDirichlet);
    b_nodes = b_nodes(freeNeumann);
    b_elements = b_elements(freeDirichlet);
    
    %*** compute Galerkin solution for coarse mesh
    x = [ W K' ; -K V ] \ [ b_nodes ; b_elements ];
    
    %*** compute coefficient vectors w.r.t. S1(GammaN) and P0(GammaD)
    xN=zeros(nC,1);
    %*** dof on Neumann boundary
    xN(freeNeumann) = x(1:nN);
    %*** dof on Dirichlet boundary
    xD = x((1:nD) + nN);  

    %*** compute error estimates
    [rhoD,rhoN] = computeEstMixResidual(vertices,coordinates,triangles, ...
        dirichlet,neumann,xN,xD,gDh,phiNh,fh,lambdah);
    rhoV =computeEstSlpResidual(vertices,triangles,elements,lambdah,[],-fh);
    rhoD = rhoD(1:nD);
    rhoN = rhoN(nD+1:end);
    rhoVD = rhoV(1:nD);
    rhoVN = rhoV(nD+1:end);
      
    %*** the following lines are only used for visualization and are not
    %*** part of the adaptive algorithm
    err(j)=sqrt(sum(computeErrDirichlet(coordinates,neumann,xN+gDh,@g))...
            +sum(computeErrNeumann(coordinates,dirichlet,...
               xD ,@phi)));
    est(j)=sqrt(sum(rhoD)+sum(rhoN)+sum(rhoV));
    oscillations(j)=sqrt(sum(oscD)+sum(oscN));
    oscillationsVol(j)=sqrt(sum(oscV));
    nE(j)=size(dirichlet,1)+size(neumann,1);
    
    %*** mark elements for refinement
    [marked_dirichlet,marked_neumann,marked_triangles] ...
        = markElements(theta,rhoD + rhoVD + oscD,rhoN + rhoVN + oscN ,oscV);
    
    %*** generate new mesh
    [vertices_new,triangles_new,dirichlet_new,neumann_new,...
        father2triangles,father2dirichlet,father2neumann_adap] ...
        = refineMesh(vertices,triangles,dirichlet,neumann,...
        marked_triangles, marked_dirichlet,marked_neumann);
             
    %*** stopping criterion
    if (size(neumann_new,1) + size(dirichlet_new,1)...
            +size(triangles_new,1) > nEmax )
        break;
    else
        j=j+1;
    end
    vertices=vertices_new;
    triangles=triangles_new;
    dirichlet=dirichlet_new;
    neumann=neumann_new;
             
end

%*** the user should read the help section of this demo
disp(' ');
disp('Please read the documentation of this demo for details on the ');
disp('adaptive loop and on the problem considered.');
disp(['To that end, type ''help ',mfilename,''' at the MATLAB prompt.']);

%*** visualize error, estimator, and oscillations
figure(1);
clf;
loglog(nE,err,'--x',nE,est,'--o',nE,oscillations,'-->',...
    nE,oscillationsVol,'--<',nE,nE.^(-3/2),'--k');
title('mixed model problem with residual-error estimator');
xlabel('number of elements');
ylabel('error bound, estimator, and oscillations');
legend('error','rho','osc','volume osc','O(N^{-3/2})');
axis tight

%*** visualize exact and adaptively computed solution
figure(2);
clf;
plotArclengthS1(coordinates,[dirichlet;neumann],...
    xN+gDh,@g);
title('BEM solution vs. exact solution: u');
xlabel('arclength from origin');
legend('BEM solution','exact solution');

figure(3);
clf;
phih = [xD ; phiNh(nD+1:end)];
plotArclengthP0(coordinates,[dirichlet;neumann],phih,@phi);
title('BEM solution vs. exact solution: phi');
xlabel('arclength from origin');
legend('BEM solution','exact solution');
