%ADAPTIVEHYPVOLRESIDUAL   Demo for hypsing IE with rho estimator.
%   ADAPTIVEHYPVOLRESIDUAL provides the implementation of an adaptive mesh-
%   refining algorithm for the hypersingular integral equation with volume 
%   force, steered by the residual-type error estimator rho. We consider
%
%      W*u = (K'-1/2)*phiN - N_1 f
%
%   which is equivalent to the Neumann problem
% 
%      -Laplace(u) = f in Omega with du/dn = phiN on Gamma
%
%   As example serves an L-shaped domain Omega with known exact solution
%
%      u(x) = |x-z|^(1.8) + r^(2/3)*cos(2 \theta / 3)
%
%   with z = [0.14 0.14], prescribed in polar coordinates.
%   The exact solution u exhibits a generic singularity. Given an initial
%   mesh, the adaptive algorithm reads as follows:
%
%   (1) build Galerkin data w.r.t. (E,T) and compute data oscillations
%   (2) compute Galerkin solution w.r.t. (E,T)
%   (3) compute E-elementwise contributions of
%       rho, estimate the error induced by building the RHS
%   (4) use Doerfler strategy to mark certain
%       (E,T)-elements
%   (5) refine marked elements to obtain new mesh
%       (E,T)
%   (6) iterate as long as #E+#T is lower than
%       given bound
%
%   At the end, the error, the estimator as well as the oscillations in each
%   step of the algorithm are plotted over the number of elements. 
%   Additionally, we plot the BEM solution vs. the exact solution over the
%   arclength.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

%*** add hilberttools main directory
addpath('../../');

%*** load L-shaped domain Omega
vertices = load('coordinates.dat');
triangles = load('triangles.dat');

%*** rotate domain Omega so that exact solution is symmetric
alpha = 3*pi/4;
vertices = vertices*[cos(alpha) -sin(alpha);sin(alpha) cos(alpha)]';

%*** shrink domain to ensure ellipticity of V by diam(Omega)<1
vertices = vertices/4;

%*** maximal number of elements
nEmax = 500;

%*** adaptivity parameter 
theta = 0.25;

%*** counter
j = 1;

%*** number of elements in each step of adaptive loop
nE = zeros(2,1);

%*** save exact error, estimator, and oscillations in each step of the 
%*** adaptive loop
err = zeros(2,1);
est = zeros(2,1);
oscillations = zeros(2,1);
oscillationsVol = zeros(2,1);

%*** extract boundary mesh
[vertices,triangles,coordinates,elements] = ...
    buildSortedMesh(vertices,triangles);

%*** adaptive mesh-refining algorithm
while 1
    
    fprintf('number of elements: N = %d (Gamma), %d (Omega)\r', ...
        size(elements,1),size(triangles,1));

    %*** discretize Neumann and volume data and compute data oscillations
    [osc,phih] = computeOscNeumann(coordinates,elements,@phi);
    [oscV,fh] = computeOscVolume(vertices,triangles,@f);

    %*** compute solution
    W = buildW(coordinates,elements) ...
        + buildHypsingStabilization(coordinates,elements);
    [b,lambdah] = buildHypsingVolRHS(coordinates,elements,phih, ...
        vertices,triangles,fh);
    x = W\b;
    
    %*** compute weighted-residual error estimator rho
    rho = computeEstHypResidual(coordinates,elements,x,phih,lambdah);
    
    %*** the following lines are only used for visualization and are not
    %*** part of the adaptive algorithm
    nE(j) = size(elements,1);
    err(j) = sqrt(sum(computeErrDirichlet(coordinates,elements,x,@g)));
    est(j) = sqrt(sum(rho));
    oscillations(j) = sqrt(sum(osc));
    oscillationsVol(j) = sqrt(sum(oscV));
    
    %*** compute weighted-residual estimator rho_V for Symm's equation to
    %*** measure error induced within buildHypsingVolRHS
    rho_V_ = computeEstSlpResidual(vertices,triangles,elements,lambdah,[],-fh);

    %*** mark elements for refinement
    [marked_elements,marked_triangles] = markElements(theta, ...
        rho + rho_V_ + osc,oscV);

    %*** generate new mesh
    [vertices,triangles,elements_new] ...
        = refineMesh(vertices,triangles,elements,marked_triangles,marked_elements);
    coordinates_new = vertices(unique(elements_new),:);

    %*** stopping criterion
    if (size(elements_new,1) + size(triangles,1)) > nEmax
      break;
    else
      j = j+1;
    end
    
    coordinates = coordinates_new;
    elements = elements_new;
end

%*** the user should read the help section of this demo
disp(' ');
disp('Please read the documentation of this demo for details on the ');
disp('adaptive loop and on the problem considered.');
disp(['To that end, type ''help ',mfilename,''' at the MATLAB prompt.']);

%*** visualize error, estimator, and oscillations
figure(1);
clf;
loglog(nE,err,'--x',nE,est,'--o',nE,oscillations,'-->',nE,oscillationsVol,'--<',nE,nE.^(-3/2),'--k');
title('Symms integral equation with residual-type error estimator');
xlabel('number of boundary elements N');
ylabel('error bound, estimator, and oscillations');
legend('error','rho','osc','volume osc','O(N^{-3/2})');
axis tight

%*** visualize exact and adaptively computed solution
figure(2);
clf;
plotArclengthS1(coordinates,elements,x,@g);
title('BEM solution vs. exact solution');
xlabel('arclength from origin');
legend('BEM solution','exact solution');
