%ADAPTIVEHYPVOLMUTILDE   Demo for hypsing IE with mu-tilde estimator.
%   ADAPTIVEHYPVOLMUTILDE provides the implementation of an adaptive mesh-
%   refining algorithm for hypersingular integral equation with volume 
%   force, steered by the (h-h/2)-type error estimator mu-tilde. We consider
%
%      W*u = (K'-1/2)*phiN - N_1 f
%
%   which is equivalent to the Neumann problem
% 
%      -Laplace(u) = f in Omega with du/dn = phiN on Gamma
%
%   As example serves an L-shaped domain Omega with known exact solution
%
%      u(x) = |x-z|^(1.8) + r^(2/3)*cos(2 \theta / 3)
%
%   with z = [0.14 0.14], prescribed in polar coordinates.
%   The exact solution u exhibits a generic singularity. Given an initial
%   mesh, the adaptive algorithm reads as follows:
%
%   (1) refine given boundary mesh Ecoarse uniformly and obtain
%       Efine
%   (2) build Galerkin data w.r.t. (Efine,Tcoarse) and compute data
%       oscillations
%   (3) compute Galerkin solution w.r.t. (Efine,Tcoarse)
%   (4) compute Ecoarse-elementwise contributions of
%       mu-tilde and estimate the error induced by building the RHS
%   (5) use Doerfler strategy to mark certain
%       (Ecoarse,Tcoarse)-elements
%   (6) refine marked elements to obtain new mesh
%       (Ecoarse,Tcoarse)
%   (7) iterate as long as #Ecoarse+#Tcoarse is lower than
%       given bound
%
%   At the end, the error, the estimator as well as the oscillations in each
%   step of the algorithm are plotted over the number of boundary elements. 
%   Additionally, we plot the BEM solution vs. the exact solution over the
%   arclength.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

%*** add hilberttools main directory
addpath('../../'); 

%*** load L-shaped domain Omega
vertices = load('coordinates.dat');
triangles = load('triangles.dat');

%*** rotate domain Omega so that exact solution is symmetric
alpha = 3*pi/4;
vertices = vertices*[cos(alpha) -sin(alpha);sin(alpha) cos(alpha)]';

%*** shrink domain to ensure ellipticity of V by diam(Omega)<1
vertices = vertices/4;

%*** maximal number of elements
nEmax = 500;

%*** adaptivity parameter
theta = 0.25;

%*** counter
j = 1;

%*** store number of elements, exact error, error estimator, and data
%*** oscillations in each step of the adaptive loop
nE = zeros(2,1);
err = zeros(2,1);
est = zeros(2,1);
oscillations = zeros(2,1);
oscillationsVol = zeros(2,1);

%*** extract boundary mesh
[vertices,triangles,coordinates,elements] = ...
    buildSortedMesh(vertices,triangles);

%*** adaptive mesh-refining algorithm
while 1
    
    fprintf('number of elements: N = %d (Gamma), %d (Omega)\r', ...
        size(elements,1),size(triangles,1));

    %*** build uniformly refined mesh
    [coordinates_fine,elements_fine,father2son] ...
        = refineBoundaryMesh(coordinates,elements);
    
    %*** discretize Neumann and volume data and compute data oscillations
    [osc_fine,phih_fine] ...
        = computeOscNeumann(coordinates_fine,elements_fine,@phi);
    [oscV,fh] = computeOscVolume(vertices,triangles,@f);
    osc = osc_fine(father2son(:,1)) + osc_fine(father2son(:,2));

    %*** compute fine-mesh solution
    W_fine = buildW(coordinates_fine,elements_fine) ...
             + buildHypsingStabilization(coordinates_fine,elements_fine);
    [b_fine,y_fine] = buildHypsingVolRHS(coordinates_fine,elements_fine,phih_fine, ...
        vertices,triangles,fh);
    x_fine = W_fine\b_fine;

    %*** compute (h-h/2)-error estimator mu-tilde
    mu_tilde = computeEstHypMuTilde(elements_fine,elements,father2son,x_fine);

    %*** the following lines are only used for visualization and are not
    %*** part of the adaptive algorithm
    nE(j) = size(elements,1);
    W = buildW(coordinates,elements) ...
        + buildHypsingStabilization(coordinates,elements);
    [~, phih] = computeOscNeumann(coordinates,elements,@phi);
    b = buildHypsingVolRHS(coordinates,elements,phih,vertices,triangles,fh);
    x = W\b;
    err(j) = sqrt(sum(computeErrDirichlet(coordinates,elements,x,@g)));
    est(j) = sqrt(sum(mu_tilde));
    oscillations(j) = sqrt(sum(osc));
    oscillationsVol(j) = sqrt(sum(oscV));
    
    %*** compute (h-h/2)-error estimator mu-tilde for Symm's equation to
    %*** measure error induced within buildHypsingVolRHS
    mu_V_tilde_ = computeEstSlpMuTilde(coordinates,elements,father2son,y_fine);

    %*** mark elements for refinement
    [marked_elements,marked_triangles] = markElements(theta, ...
        mu_tilde + mu_V_tilde_ + osc,oscV);

    %*** generate new mesh
    [vertices,triangles,elements] ...
        = refineMesh(vertices,triangles, ...
	    elements,marked_triangles,marked_elements);
    coordinates = vertices(unique(elements),:);

    %*** stopping criterion
    if (size(elements,1) + size(triangles,1)) > nEmax
      break;
    else
      j = j+1;
    end
end

%*** the user should read the help section of this demo
disp(' ');
disp('Please read the documentation of this demo for details on the ');
disp('adaptive loop and on the problem considered.');
disp(['To that end, type ''help ',mfilename,''' at the MATLAB prompt.']);

%*** visualize error, estimator, and oscillations
figure(1);
clf;
loglog(nE,err,'--x',nE,est,'--o',nE,oscillations,'-->', ...
    nE,oscillationsVol,'--<',nE,nE.^(-3/2),'--k');
xlabel('number of boundary elements N');
ylabel('error bound, estimator, and oscillations');
legend('error','mu-tilde','osc','volume osc','O(N^{-3/2})');
axis tight

%*** visualize exact and adaptively computed solution
figure(2);
clf;
plotArclengthS1(coordinates_fine,elements_fine,x_fine,@g);
title('BEM solution vs. exact solution');
xlabel('arclength from origin');
legend('BEM solution','exact solution');
