%ADAPTIVESYMMRESIDUALINDIRECT   Demo for Symm's IE with rho estimator.
%   ADAPTIVESYMMRESIDUALINDIRECT provides the implementation of an adaptive
%   mesh-refining algorithm for Symm's integral equation, steered by the
%   residual-type error estimator rho on a slit. We consider
%
%      V*phi = g.
%
%   We use functionality provided by the HILBERT program package. As example
%   serves the so-called slit-problem, where Gamma = (-1,1) x {0} and
%   g(x,y) = -x. The exact solution phi is then known explicitly,
%
%      phi(x,y) = -2*x/(1-x^2)^{1/2}.
%
%   which exhibits generic singularities at the endpoints of Gamma.
%   Given an initial mesh, the adaptive algorithm reads as follows:
%
%   (1) compute Galerkin solution w.r.t. E
%   (2) compute E-elementwise contributions of rho and
%       Dirichlet data oscillations
%   (3) use Doerfler strategy to mark certain E-elements
%   (4) refine marked elements to obtain new mesh E
%   (5) iterate as long as #E is lower than given bound
%
%   At the end, the error, and the estimator in each
%   step of the algorithm are plotted over the number of elements. 
%   Additionally, we plot the BEM solution vs. the exact solution over the
%   arclength.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

%*** add hilberttools main directory
addpath('../../');

%*** define computational domain
coordinates = [-1 0; 1 0];
elements = [1 2];

%*** define function handle for g(x,y) = -x
g_s = @(x) -x(:,1);

%*** define exact solution phi
phi_s = @(x,a,b) -2*x(:,1)./sqrt(1-x(:,1).^2);

%*** maximal number of elements
nEmax = 200;

%*** adaptivity parameter 
theta = 0.25;
theta_min = 0.25;

%*** counter
j = 1;

%*** store number of elements, exact error, and error estimator in each 
%*** step of the adaptive loop
nE = zeros(2,1);
err = zeros(2,1);
est = zeros(2,1);

%*** adaptive mesh-refining algorithm
while 1
    
    fprintf('number of elements: N = %d\r',size(elements,1))

    %*** discretize Dirichlet data. Oscillations are not stored since they
    %*** are zero due to the linear RHS
    [~,gh] = computeOscDirichlet(coordinates,elements,g_s);

    %*** compute solution
    V = buildV(coordinates,elements);
    M = buildM(coordinates,elements);
    b = M*gh;
    x = V\b;
    
    %*** compute weighted residual-error estimator rho
    rho = computeEstSlpResidual(coordinates,elements,x,[],gh);

    %*** the following lines are only used for visualization and are not
    %*** part of the adaptive algorithm
    nE(j) = size(elements,1);
    err(j) = sqrt(sum(computeErrNeumann(coordinates,elements,x,phi_s)));
    est(j) = sqrt(sum(rho));
    
    %*** mark elements for refinement
    marked = markElements(theta,theta_min,rho);

    %*** generate new mesh
    [coordinates_new,elements_new] = refineBoundaryMesh( ...
      coordinates,elements,marked);
    
    if size(elements_new,1) > nEmax
      break;
    else
      j = j+1;
    end
    coordinates = coordinates_new;
    elements = elements_new;
end

%*** the user should read the help section of this demo
disp(' ');
disp('Please read the documentation of this demo for details on the ');
disp('adaptive loop and on the problem considered.');
disp(['To that end, type ''help ',mfilename,''' at the MATLAB prompt.']);

%*** visualize error and estimator
figure(1);
clf;
loglog(nE,err,'--x',nE,est,'--o',nE,nE.^(-3/2),'--k');
title('Symms integral equation with residual-type error estimator');
xlabel('number of elements');
ylabel('error bound, and estimator');
legend('error','rho','O(N^{-3/2})');
axis tight

%*** visualize exact and adaptively computed solution
figure(2);
clf;
plotArclengthP0(coordinates,elements,x,phi_s);
title('BEM solution vs. exact solution');
xlabel('arclength from origin');
legend('BEM solution','exact solution');
