%ADAPTIVESYMMHIERARCHICAL   Demo for Symm's IE with tau estimator on Lshape.
%   ADAPTIVESYMMHIERARCHICAL provides the implementation of an adaptive
%   mesh-refining algorithm for Symm's integral equation, steered by
%   the two-level error estimator tau. We consider
%
%      V*phi = (K+1/2)*g
%
%   which is equivalent to the Dirichlet problem
% 
%      -Laplace(u) = 0 in Omega with u = g on Gamma
%
%   We use functionality provided by the HILBERT program package. As example
%   serves an L-shaped domain Omega with known exact solution
%
%      u(r,theta) = r^{2/3} * cos(2theta/3)
%
%   prescribed in 2D polar coordinates, which exhibits a generic
%   singularity at the origin. Given an initial mesh, the adaptive algorithm
%   reads as follows:
%
%   (1) initialize mesh Ecoarse
%   (2) build Galerkin data w.r.t. Ecoarse
%   (3) compute Galerkin solution w.r.t. Ecoarse
%   (4) compute Ecoarse-elementwise contributions of two-level
%       error estimator tau and Dirichlet data oscillations
%       (To that end build integral operator and RHS w.r.t.
%       the uniformly refined mesh Efine)
%   (5) use Doerfler strategy to mark certain Ecoarse-elements
%   (6) refine marked elements to obtain new mesh Ecoarse
%   (7) iterate as long as #Ecoarse is lower than given bound
%
%   At the end, the error, the estimator as well as the oscillations in each
%   step of the algorithm are plotted over the number of elements. 
%   Additionally, we plot the BEM solution vs. the exact solution over the
%   arclength.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

%*** add hilberttools main directory
addpath('../../');

%*** load L-shaped domain Omega
coordinates = load('coordinates.dat');
elements = load('elements.dat');

%*** rotate domain Omega so that exact solution is symmetric
alpha = 3*pi/4;
coordinates = coordinates*[cos(alpha) -sin(alpha);sin(alpha) cos(alpha)]';

%*** shrink domain to ensure ellipticity of V by diam(Omega)<1
coordinates = coordinates/4;

%*** maximal number of elements
nEmax = 200;

%*** adaptivity parameter 
theta = 0.25;
theta_min = 0.25;

%*** counter
j = 1;

%*** store number of elements, exact error, error estimator, and Neumann
%*** data oscillations in each step of the adaptive loop
nE = zeros(2,1);
err = zeros(2,1);
est = zeros(2,1);
oscillations = zeros(2,1);

%*** adaptive mesh-refining algorithm
while 1

    fprintf('number of elements: N = %d\r',size(elements,1));

    %*** buld uniformly refined mesh
    [coordinates_fine,elements_fine,father2son] ...
        = refineBoundaryMesh(coordinates,elements);

    %*** discretize Dirichlet data and compute data oscillations
    [osc_fine,gh_fine] ...
        = computeOscDirichlet(coordinates_fine,elements_fine,@g);
    osc = osc_fine(father2son(:,1)) + osc_fine(father2son(:,2));

    %*** compute integral operators
    %fine mesh
    V_fine = buildV(coordinates_fine,elements_fine);
    %coarse mesh
    V = buildV(coordinates,elements);
    
    %*** compute right-hand side
    %fine mesh
    b_fine = buildSymmRHS(coordinates_fine,elements_fine,gh_fine);
    
    %*** obtain coarse-mesh RHS from fine-mesh RHS
    b = b_fine(father2son(:,1))+b_fine(father2son(:,2));

    %*** compute Galerkin solution for coarse mesh
    x = V\b;

    %*** compute two-level error estimator tau
    tau = computeEstSlpTau(father2son,V_fine,b_fine,x);

    %*** the following lines are only used for visualization and are not
    %*** part of the adaptive algorithm
    nE(j) = size(elements,1);
    err(j) = sqrt(sum(computeErrNeumann(coordinates,elements,x,@phi)));
    est(j) = sqrt(sum(tau));
    oscillations(j) = sqrt(sum(osc));
    
    %*** mark elements for refinement
    [marked] = markElements(theta,theta_min,tau + osc);
    
    %*** generate new mesh
    [coordinates_new,elements_new] ...
        = refineBoundaryMesh(coordinates,elements,marked);

    %*** stopping criterion + size(triangles,1)
    if (size(elements_new,1)  > nEmax )
        break;
    else
        j = j+1;
    end

    coordinates = coordinates_new;
    elements = elements_new;
end

%*** the user should read the help section of this demo
disp(' ');
disp('Please read the documentation of this demo for details on the ');
disp('adaptive loop and on the problem considered.');
disp(['To that end, type ''help ',mfilename,''' at the MATLAB prompt.']);

%*** visualize error, estimator, and oscillations
figure(1);
clf;
loglog(nE,err,'--x',nE,est,'--o',nE,oscillations,'-->',nE,nE.^(-3/2),'--k');
title('Symms integral equation with two-level error estimator');
xlabel('number of elements');
ylabel('error bound, estimator and oscillations');
legend('error','tau','osc','O(N^{-3/2})');
axis tight

%*** visualize exact and adaptively computed solution
figure(2);
clf;
plotArclengthP0(coordinates,elements,x,@phi);
title('BEM solution vs. exact solution');
xlabel('arclength from origin');
legend('BEM solution','exact solution');
