%ADAPTIVEMIXEDMUTILDE   Demo for the mixed problem with mu-tilde estimator.
%   ADAPTIVEMIXEDMUTILDE provides the implementation of an adaptive mesh-
%   refining algorithm for the mixed model problem on the L-shape, steered 
%   by the (h-h/2)-type error estimator mu-tilde. We consider
%
%      -K*gN + V*phiD = (1/2+K)*gD - V*phiN 
%      W*gN + K'*phiD = -W*gD +(1/2-K')*phiN  
%
%   which is equivalent to the mixed boundary value problem
%
%      -Laplace(u)  = 0    in Omega
%               u   = gD   on GammaD
%             du/dn = phiN on GammaN
%
%   We use functionality provided by the HILBERT program package. As example
%   serves an L-shaped domain Omega with known exact solution
%
%      u(x) = r^{2/3} * cos(2theta/3)
%
%   prescribed in 2D polar coordinates, which exhibits a generic 
%   singularity at the origin. Given an initial mesh, the adaptive algorithm
%   reads as follows:
%
%   (1) refine given mesh Ecoarse uniformly and obtain Efine
%   (2) build Galerkin data and compute Dirichlet/Neumann data oscillations
%       w.r.t. Efine
%   (3) compute Galerkin solution w.r.t. Efine
%   (4) compute Ecoarse-elementwise contributions of mu-tilde on GammaD and
%       on GammaN
%   (5) use Doerfler strategy to mark certain Ecoarse-elements
%   (6) refine marked elements to obtain new mesh Ecoarse
%   (7) iterate as long as #Ecoarse is lower than given bound
%
%   At the end, the error, the estimator as well as the oscillations in each
%   step of the algorithm are plotted over the number of elements. 
%   Additionally, we plot the BEM solution vs. the exact solution over the
%   arclength.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

%*** add hilberttools main directory
addpath('../../'); 

%*** load L-shaped domain Omega
coordinates = load('coordinates.dat');
elements = load('elements.dat');

%*** split Gamma into Dirichlet and Neumann boundary
dirichlet = [1 2;2 3;3 6;6 5];
neumann = [5 8;8 7;7 4;4 1];

%*** rotate domain Omega so that exact solution is symmetric
alpha = 3*pi/4;
coordinates = coordinates*[cos(alpha) -sin(alpha);sin(alpha) cos(alpha)]';

%*** shrink domain to ensure ellipticity of V by diam(Omega)<1
coordinates = coordinates/4;

%*** maximal number of elements
nEmax = 200;

%*** adaptivity parameter
theta = 0.25;
theta_min = 0.25;

%*** counter
j = 1;

%*** store number of elements, exact error, error estimator, and data 
%*** oscillations in each step of the adaptive loop
nE = zeros(2,1);
err = zeros(2,1);
est = zeros(2,1);
oscillations = zeros(2,1);

%*** rearrange indices such that Neumann nodes are first
[coordinates,neumann,dirichlet] = ...
    buildSortedMesh(coordinates,neumann,dirichlet);

%*** Perform one uniform refinement before starting the loop and compute
%*** data oscillations
[coordinates_fine,dirichlet_fine,neumann_fine,...
    father2dirichlet_fine,father2neumann_fine] ...
    = refineBoundaryMesh(coordinates,dirichlet,neumann);

%*** rearrange indices such that Neumann nodes are first
[coordinates_fine,neumann_fine,dirichlet_fine] = ...
    buildSortedMesh(coordinates_fine,neumann_fine,dirichlet_fine);

%*** discretize data and compute data oscillations for fine mesh
[~,~,gDh,phiNh] = computeOscMixed(coordinates,dirichlet,neumann,@g,@phi);
[oscD_fine,oscN_fine,gDh_fine,phiNh_fine] ...
    = computeOscMixed(coordinates_fine,dirichlet_fine,neumann_fine, ...
    father2neumann_fine,neumann,gDh,@g,@phi);    

oscD = sum(oscD_fine(father2dirichlet_fine),2);
oscN = sum(oscN_fine(father2neumann_fine),2);

%*** adaptive mesh-refining algorithm
while 1

    fprintf('number of elements: N = %d\r', size(dirichlet,1)+size(neumann,1));
    
    %*** compute degrees of freedom for fine mesh
    nC_fine = size(coordinates_fine,1);
    nD_fine = size(dirichlet_fine,1);
    freeNeumann_fine = setdiff(1:nC_fine,unique(dirichlet_fine));
    freeDirichlet_fine = 1:nD_fine;
    nN_fine = length(freeNeumann_fine);
    
    %*** compute integral operators for fine mesh
    elements_fine = [dirichlet_fine;neumann_fine];
    V_fine = buildV(coordinates_fine,elements_fine);
    K_fine = buildK(coordinates_fine,elements_fine);  
    W_fine = buildW(coordinates_fine,elements_fine);

    %*** compute right-hand side for fine mesh
    [b_nodes_fine,b_elements_fine] ... 
        = buildMixedRHS(coordinates_fine,dirichlet_fine, ...
        neumann_fine,V_fine,K_fine,W_fine,gDh_fine,phiNh_fine);
    
    %*** shrink integral operators and right-hand side
    W_fine = W_fine(freeNeumann_fine,freeNeumann_fine);
    K_fine = K_fine(freeDirichlet_fine,freeNeumann_fine);
    V_fine = V_fine(freeDirichlet_fine,freeDirichlet_fine);
    b_nodes_fine = b_nodes_fine(freeNeumann_fine);
    b_elements_fine = b_elements_fine(freeDirichlet_fine);

    %*** compute Galerkin solution for fine mesh
    x = [W_fine K_fine' ; -K_fine V_fine] \...
        [b_nodes_fine ; b_elements_fine];

    %*** compute coefficient vectors w.r.t. S1(GammaN) and P0(GammaD)
    xN_fine = zeros(nC_fine,1);
    %*** dof on Neumann boundary
    xN_fine(freeNeumann_fine) = x(1:nN_fine); 
    %*** dof on Dirichlet boundary
    xD_fine = x((1:nD_fine) + nN_fine);           

    %*** compute (h-h/2)-error estimator tilde-mu on GammaD and GammaN
    muD_tilde = computeEstSlpMuTilde(coordinates,dirichlet, ...
        father2dirichlet_fine,xD_fine);
    muN_tilde = computeEstHypMuTilde(neumann_fine,neumann, ...
        father2neumann_fine,xN_fine);

    %*** the following lines are only used for visualization and are not
    %*** part of the adaptive algorithm
    nE(j)=size(dirichlet,1)+size(neumann,1);
    elements = [dirichlet;neumann];
    V = buildV(coordinates,elements);
    K = buildK(coordinates,elements);  
    W = buildW(coordinates,elements);
    [oscDcoarse,oscNcoarse,gDh,phiNh] ...
        = computeOscMixed(coordinates,dirichlet,neumann,@g,@phi);    
    [b_nodes,b_elements] ... 
        = buildMixedRHS(coordinates,dirichlet, ...
        neumann,V,K,W,gDh,phiNh);
    nC = size(coordinates,1);
    nD = size(dirichlet,1);
    freeNeumann = setdiff(1:nC,unique(dirichlet));
    nN=length(freeNeumann);
    freeDirichlet = 1:nD;
    W = W(freeNeumann,freeNeumann);
    K = K(freeDirichlet,freeNeumann);
    V = V(freeDirichlet,freeDirichlet);
    b_nodes = b_nodes(freeNeumann);
    b_elements = b_elements(freeDirichlet);
    x = [ W K' ; -K V ] \ [ b_nodes ; b_elements ];
    xN = zeros(nC,1);
    xN(freeNeumann) = x(1:nN);
    xD = x((1:nD) + nN);  
    err(j)=sqrt(sum(computeErrDirichlet(coordinates,neumann,xN+gDh,@g))...
            +sum(computeErrNeumann(coordinates,dirichlet,...
               xD ,@phi)));
    est(j)=sqrt(sum(muD_tilde)+sum(muN_tilde));
    oscillations(j)=sqrt(sum(oscDcoarse)+sum(oscNcoarse));
    
    %*** mark elements for refinement
    [marked_dirichlet,marked_neumann] ...
        = markElements(theta,theta_min,muD_tilde + oscD,muN_tilde + oscN);
    
    %*** generate new mesh
    [coordinates_new,dirichlet_new,neumann_new,father2dirichlet_adap,...
        father2neumann_adap] = refineBoundaryMesh(coordinates,dirichlet,...
        neumann,marked_dirichlet,marked_neumann);

    %*** stopping criterion
    if (size(neumann_new,1) + size(dirichlet_new,1) > nEmax )
        break;
    else
        j=j+1;
    end
    coordinates=coordinates_new;
    dirichlet=dirichlet_new;
    neumann=neumann_new;
    
    %*** rearrange indices such that Neumann nodes are first
    [coordinates,neumann,dirichlet] = ...
        buildSortedMesh(coordinates,neumann,dirichlet);

    neumann_coarse = neumann_fine;
    father2neumann_coarse = father2neumann_fine;
    [coordinates_fine,dirichlet_fine,neumann_fine,father2dirichlet_fine,...
        father2neumann_fine]= refineBoundaryMesh(...
        coordinates,dirichlet,neumann);

    %*** build coarse2fine array which relates the uniform refinement from
    %*** previous step with the current uniform refinement
    coarse2fine = generateFather2Son(father2neumann_adap, ...
                                     father2neumann_coarse, ...
                                     father2neumann_fine);

    %*** rearrange indices such that Neumann nodes are first
    [coordinates_fine,neumann_fine,dirichlet_fine] = ...
        buildSortedMesh(coordinates_fine,neumann_fine,dirichlet_fine);

    %*** discretize data and compute corresponding data oscillations for
    %*** fine mesh
    [oscD_fine,oscN_fine,gDh_fine,phiNh_fine] ...
        = computeOscMixed(coordinates_fine,dirichlet_fine,neumann_fine, ...
                          coarse2fine,neumann_coarse,gDh_fine,@g,@phi);    
    oscD = sum(oscD_fine(father2dirichlet_fine),2);
    oscN = sum(oscN_fine(father2neumann_fine),2);
end

%*** the user should read the help section of this demo
disp(' ');
disp('Please read the documentation of this demo for details on the ');
disp('adaptive loop and on the problem considered.');
disp(['To that end, type ''help ',mfilename,''' at the MATLAB prompt.']);

%*** visualize error, estimator, and oscillations
figure(1);
clf;
loglog(nE,err,'--x',nE,est,'--o',nE,oscillations,'-->',nE,nE.^(-3/2),'--k');
title('mixed model problem with mu-tilde-error estimator');
xlabel('number of elements');
ylabel('error bound, estimator, and oscillations');
legend('error','mu-tilde','osc','O(N^{-3/2})');
axis tight

%*** visualize exact and adaptively computed solution
figure(2);
clf;
plotArclengthS1(coordinates_fine,[dirichlet_fine;neumann_fine],...
    xN_fine+gDh_fine,@g);
title('BEM solution vs. exact solution: u');
xlabel('arclength from origin');
legend('BEM solution','exact solution');

figure(3);
clf;
phih = [xD_fine ; phiNh_fine(nD_fine+1:end)];
plotArclengthP0(coordinates_fine,[dirichlet_fine;neumann_fine],phih,@phi);
title('BEM solution vs. exact solution: phi');
xlabel('arclength from origin');
legend('BEM solution','exact solution');
