%ADAPTIVEHYPRESIDUALINDIRECT   Demo for hypsing IE on slit with rho est.
%   ADAPTIVEHYPRESIDUALINDIRECT provides the implementation of an adaptive
%   mesh-refining algorithm for the hypersingular integral equation by use 
%   of the weighted residual error estimator rho.
%
%   This demo provides an implementation of an adaptive mesh-refining 
%   algorithm for the hypersingular integral equation
%
%                Wu = phi
%
%   As example serves the so-called slit-problem, where Gamma = (-1,1) x {0}
%   and phi(x,y) = 1. The exact solution u is then known explicitely,
%
%      u(x,y) = 2*(1-x^2)^{1/2}.
%
%   and exhibits generic singularities at the endpoints of Gamma. Given an
%   initial mesh, the adaptive algorithm implemented below reads as follows:
%
%   (1) build Galerkin data  w.r.t. E
%   (2) compute Galerkin solution w.r.t. E
%   (3) compute E-elementwise contributions of rho
%   (4) use Doerfler strategy to mark certain E-elements
%   (5) refine marked elements to obtain new mesh E
%   (6) iterate as long as #E is lower than given bound
%
%   At the end, the error and the estimator in each
%   step of the algorithm are plotted over the number of elements. 
%   Additionally, we plot the BEM solution vs. the exact solution over the
%   arclength.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

%*** add hilberttools main directory
addpath('../../');

%*** define computational domain
coordinates = [-1 0; 1 0];
elements = [1 2];

%*** uniform-refinement to obtain initial mesh
[coordinates,elements] = refineBoundaryMesh(coordinates,elements);

%*** define function handle for phi_s = 1
phi_s = @(x,a,b) ones(size(x,1),1);

%*** define function handle for exact solution u_s
u_s = @(x) 2*sqrt(1-x(:,1).^2);

%*** maximal number of elements
nEmax = 200;

%*** adaptivity parameter 
theta = 0.25;
theta_min = 0.25;

%*** counter
j = 1;

%*** store number of elements, exact error, error estimator, and Neumann
%*** data oscillations in each step of the adaptive loop
nE = zeros(2,1);
err = zeros(2,1);
est = zeros(2,1);

%*** adaptive mesh-refining algorithm
while 1
    
    fprintf('number of elements: N = %d\r',size(elements,1))

    %*** discretize Neumann data. Oscillations are not stored since they
    %*** are zero due to the constant RHS
    [~,phih] = computeOscNeumann(coordinates,elements,phi_s);

    %*** build Galerkin data
    W = buildW(coordinates,elements);
    M = buildM(coordinates,elements);
    b = M'*phih;
    
    %*** determine free boundary
    fixed=[setdiff(elements(:,1),elements(:,2)),...
        setdiff(elements(:,2),elements(:,1))];
    free = setdiff(1:size(coordinates,1),fixed);
    
    %*** compute Galerkin solution
    x_free = W(free,free)\b(free);
    x=zeros(size(coordinates,1),1);
    x(free) = x_free;
    
    %*** compute weighted residual-error estimator rho
    rho = computeEstHypResidual(coordinates,elements,x,[],phih);

    %*** the following lines are only used for visualization and are not
    %*** part of the adaptive algorithm
    nE(j) = size(elements,1);
    err(j) = sqrt(sum(computeErrDirichlet(coordinates,elements,x,u_s)));
    est(j) = sqrt(sum(rho));
    
    %*** mark elements for refinement
    marked = markElements(theta,theta_min,rho);

    %*** generate new mesh
    [coordinates_new,elements_new] = refineBoundaryMesh( ...
      coordinates,elements,marked);
    
    if size(elements_new,1) > nEmax
      break;
    else
      j = j+1;
    end
    coordinates = coordinates_new;
    elements = elements_new;
end

%*** the user should read the help section of this demo
disp(' ');
disp('Please read the documentation of this demo for details on the ');
disp('adaptive loop and on the problem considered.');
disp(['To that end, type ''help ',mfilename,''' at the MATLAB prompt.']);

%*** visualize error and estimator
figure(1);
clf;
loglog(nE,err,'--x',nE,est,'--o',nE,nE.^(-3/2),'--k');
title('Hypersingular integral equation with residual-type error estimator');
xlabel('number of elements');
ylabel('error bound, and estimator');
legend('error','rho','O(N^{-3/2})');
axis tight

%*** visualize exact and adaptively computed solution
figure(2);
clf;
plotArclengthS1(coordinates,elements,x,u_s);
title('BEM solution vs. exact solution');
xlabel('arclength from origin');
legend('BEM solution','exact solution');
