%ADAPTIVEHYPRESIDUAL   Demo for hypsing IE on Lshape with rho estimator.
%   ADAPTIVEHYPRESIDUAL provides the implementation of an adaptive mesh-
%   refining algorithm for the hypersingular integral equation, steered by
%   the residual-type error estimator rho. We consider
%
%      W*u = (K'-1/2)*phiN
%
%   which is equivalent to the Neumann problem
% 
%      -Laplace(u) = 0 in Omega with du/dn = phiN on Gamma
%
%   We use functionality provided by the HILBERT program package. As example
%   serves an L-shaped domain Omega with known exact solution
%
%      u(x) = r^{2/3} * cos(2theta/3)
%
%   prescribed in 2D polar coordinates, which exhibits a generic 
%   singularity at the origin. Given an initial mesh, the adaptive algorithm
%   reads as follows:
%
%   (1) Build Galerkin data and compute Neumann data oscillations w.r.t. E
%   (2) compute Galerkin solution w.r.t. E
%   (3) compute E-elementwise contributions of rho 
%   (4) use Doerfler strategy to mark certain E-elements
%   (5) refine marked elements to obtain new mesh E
%   (6) iterate as long as #E is lower than given bound
%
%   At the end, the error, the estimator as well as the oscillations in each
%   step of the algorithm are plotted over the number of elements. 
%   Additionally, we plot the BEM solution vs. the exact solution over the
%   arclength.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

%*** add hilberttools main directory
addpath('../../');

%*** load L-shaped domain Omega
coordinates = load('coordinates.dat');
elements = load('elements.dat');

%*** rotate domain Omega so that exact solution is symmetric
alpha = 3*pi/4;
coordinates = coordinates*[cos(alpha) -sin(alpha);sin(alpha) cos(alpha)]';

%*** shrink domain to ensure ellipticity of V by diam(Omega)<1
coordinates = coordinates/4;

%*** maximal number of elements
nEmax = 200;

%*** adaptivity parameter 
theta = 0.25;
theta_min = 0.25;

%*** counter
j = 1;

%*** store number of elements, exact error, error estimator, and Neumann
%*** data oscillations in each step of the adaptive loop
nE = zeros(2,1);
err = zeros(2,1);
est = zeros(2,1);
oscillations = zeros(2,1);

%*** adaptive mesh-refining algorithm
while 1
    
    fprintf('number of elements: N = %d\r',size(elements,1))

    %*** discretize Neumann data and compute data oscillations
    [osc,phih] = computeOscNeumann(coordinates,elements,@phi);
    
    %*** compute coarse-mesh solution
    W = buildW(coordinates,elements) ...
             + buildHypsingStabilization(coordinates,elements);
    b = buildHypsingRHS(coordinates,elements,phih);
    x = W\b;
    
    %*** compute weighted-residual-error estimator rho
    rho = computeEstHypResidual(coordinates,elements,x,phih);

    %*** the following lines are only used for visualization and are not
    %*** part of the adaptive algorithm
    nE(j) = size(elements,1);
    err(j) = sqrt(sum(computeErrDirichlet(coordinates,elements,x,@g)));
    est(j) = sqrt(sum(rho));
    oscillations(j) = sqrt(sum(osc));
    
    %*** mark elements for refinement
    marked = markElements(theta,theta_min,rho + osc);

    %*** generate new mesh
    [coordinates_new,elements_new] ...
        = refineBoundaryMesh(coordinates,elements,marked);
    
    if size(elements_new,1) > nEmax
      break;
    else
      j = j+1;
    end

    coordinates = coordinates_new;
    elements = elements_new;
end

%*** the user should read the help section of this demo
disp(' ');
disp('Please read the documentation of this demo for details on the ');
disp('adaptive loop and on the problem considered.');
disp(['To that end, type ''help ',mfilename,''' at the MATLAB prompt.']);

%*** visualize error, estimator, and oscillations
figure(1);
clf;
loglog(nE,err,'--x',nE,est,'--o',nE,oscillations,'-->',nE,nE.^(-3/2),'--k');
title('Hypersingular integral equation with residual-type error estimator');
xlabel('number of elements');
ylabel('error bound, estimator, and oscillations');
legend('error','rho','osc','O(N^{-3/2})');
axis tight

%*** visualize exact and adaptively computed solution
figure(2);
clf;
plotArclengthS1(coordinates,elements,x,@g);
title('BEM solution vs. exact solution');
xlabel('arclength from origin');
legend('BEM solution','exact solution');
