function [osc,fh] = computeOscVolume(vertices,triangles,f)
%COMPUTEOSCVOLUME   Computes oscillations and discrete volume data.
%   [OSC,FH] = COMPUTEOSCVOLUME(VERTICES,TRIANGLES,F) discretizes the given
%   volume data by L2 projection onto piecewise constants and computes the 
%   corresponding volume data oscillations.
%
%   For a given partition {T1,...,TN} of the body Omega into triangles Tj, 
%   the function returns the (N x 1)-vector OSC, where 
%
%      OSC(j) = area(Tj) * || f - f_Tj ||_{L2(Tj)}^2
%
%   Here, f_Tj denotes the integral mean of f over Tj. The L2-norm as well 
%   as the integral mean are computed via a Gaussian quadrature rule with 7
%   vertices which is exact on polynomials in R^2 of absolute degree 5. In 
%   addition the function returns the (N x 1)-vector of the elementwise 
%   values FH(j) = f_Tj.
%
%   The (K x 2)-matrix VERTICES provides the coordinates for all vertices of
%   the partition, whereas the (N x 3)-matrix TRIANGLES provides the indices
%   of the respective vertices for all all triangles Tj. F is a function 
%   handle for the volume force. For an (M x 2)-matrix of nodes X, it 
%   returns the (M x 1)-vector Y of corresponding function values, i.e. 
%   Y(k) := F(X(k,:)).

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

%*** quadrature rule on reference triangle Tref = conv{(0,0),(1,0),(0,1)}
tmp_pos = [6-sqrt(15) ; 9+2*sqrt(15) ; 6+sqrt(15) ; 9-2*sqrt(15) ; 7]/21;
quad_vertices = tmp_pos([1 1 ; 2 1 ; 1 2 ; 3 4 ; 3 3 ; 4 3 ; 5 5]);
tmp_wts = [155-sqrt(15) 155+sqrt(15) 270]/2400;
quad_weights = tmp_wts([1 1 1 2 2 2 3]);

%*** the remaining code is independent of the chosen quadrature rule
nT = size(triangles,1);
nQ = size(quad_vertices,1);

%*** first vertices of triangles and corresponding edge vectors
v1  = vertices(triangles(:,1),:);
d21 = vertices(triangles(:,2),:) - v1;
d31 = vertices(triangles(:,3),:) - v1;

%*** compute vector of triangle areas 2*area(T)
area2 = d21(:,1).*d31(:,2)-d21(:,2).*d31(:,1);

%*** build matrix of quadrature vertices by use of affine transformation of
%*** Tref
jacobian1 = reshape(repmat([d21(:,1);d31(:,1)],1,nQ)',nT*nQ,2);
jacobian2 = reshape(repmat([d21(:,2);d31(:,2)],1,nQ)',nT*nQ,2);
zref = repmat(quad_vertices,nT,1);
z = [ sum(zref.*jacobian1,2) sum(zref.*jacobian2,2) ] ...
    + reshape(repmat(v1(:),1,nQ)',nT*nQ,2);

%*** evaluate volume force f at all quadrature vertices z
fz = reshape(f(z),nQ,nT);

%*** compute integral mean
f_mean = 2*quad_weights*fz;

%*** return OSC(j) = area(Tj) *  || f - f_mean ||_{L2(Tj)}^2
osc = 0.5*area2.^2 .* (quad_weights*(fz - repmat(f_mean,nQ,1)).^2)';

%*** return column vector
fh = f_mean';
