function [oscD,oscN,uDh,phiNh] = computeOscMixed(coordinates,dirichlet, ...
                                   neumann, varargin)
%COMPUTEOSCMIXED   Computes oscillations and discrete data.
%   [OSCD,OSCN,UDH,PHINH] = COMPUTEOSCMIXED(COORDINATES,NEUMANN, ...
%                  DIRICHLET,[FATHER2NEUMANN,NEUMANN_OLD,UDH_OLD,] UD, PHIN)
%   discretizes the given Dirichlet and Neumann data by nodal interpolation
%   and L2-projection onto piecewise constants, respectively, and returns 
%   the corresponding data oscillations.
%
%   When solving the mixed boundary value problem
%
%      -Laplace(u)  = f    in Omega
%               u   = uD   on GammaD
%             du/dn = phiN on GammaN
%
%   the given data uD and phiN have to be extended to the entire boundary 
%   Gamma. For the Dirichlet data uD, this is done by setting 
%
%      uDTilde(zj) = 0
%
%   for all nodes zj of some initial partition which lie solely on the 
%   Neumann boundary GammaN. Then, uDTilde on closure(GammaN) is given by 
%   affine interpolation. The Neumann data phiN are extended to phiNTilde by
%   zero on GammaD. In each step of the adaptive loop, it is essential to 
%   deal with the same extensions.
%
%   For the numerical computations, uDTilde is replaced by its nodal 
%   interpolant uDh, whereas phiNTilde is replaced by its L2-projection onto
%   piecewise constants. Since these extensions are only implicitly built, 
%   it is essential that the nodal vector of uDh on GammaN is appropriately
%   prolongated.
%
%   COORDINATES gives the coordinates for all nodes on the boundary. 
%   DIRICHLET and NEUMANN specify the boundary elements on the Dirichlet 
%   boundary GammaD and the Neumann boundary GammaN. UD and PHIN are 
%   function handles providing the Dirichlet data uD and the Neumann data 
%   phiN. For an (M x 2)-matrix of nodes X, UD returns the (M x 1)-vector Y
%   of corresponding function values, i.e. Y(j) := uD(X(j,:)). For an 
%   (M x 2)-matrix of nodes X and the respective boundary elements given by
%   the (M x 2) matrices A and B, PHIN returns the (M x 1)-vector Y of 
%   corresponding function values, i.e. Y(j) := PHIN(X(j,:)).
%
%   If the mesh described in terms of COORDINATES, DIRICHLET, and NEUMANN is
%   obtained by refinement, NEUMANN_OLD specifies the boundary elements on 
%   GammaN before refinement. The link between NEUMANN and NEUMANN_OLD is 
%   provided by FATHER2NEUMANN. The column vector UDH_OLD with the nodal 
%   values of uDTilde on the prior mesh is use to obtain the correct values
%   of uDTilde on the refined Neumann boundary.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

uDh = zeros(size(coordinates,1),1);
if nargin == 5
    %*** if current mesh is the initial mesh
    [uD,phiN] = varargin{:};
else
    %*** if current mesh is obtained by refinement
    [father2neumann,neumann_old,uDh_old,uD,phiN] = varargin{:};

    %*** prolongation of uDh_old to uDh on Neumann boundary GammaN
    uDh(neumann(father2neumann(:,1),2)) = 0.5*sum(uDh_old(neumann_old),2);
    uDh(neumann(father2neumann(:,1),1)) = uDh_old(neumann_old(:,1));
    uDh(neumann(father2neumann(:,2),2)) = uDh_old(neumann_old(:,2));
end

%*** discretize Dirichlet data and compute data oscillations on GammaD
[oscD,uDh_dirichlet] = computeOscDirichlet(coordinates,dirichlet,uD);

%*** update uDh on Dirichlet boundary GammaD
idx = unique(dirichlet);
uDh(idx) = uDh_dirichlet(idx);

%*** discretize Neumann data and compute data oscillations on GammaN
[oscN,phiNh] = computeOscNeumann(coordinates,neumann,phiN);

%*** prolongate coefficient vector from GammaN to entire boundary Gamma
phiNh = [zeros(size(dirichlet,1),1) ; phiNh];

