function [osc,uDh] = computeOscDirichletL2(coordinates,elements,uD,varargin)
%COMPUTEOSCDIRICHLETL2   Computes oscillations and discrete Dirichlet data.
%   [OSC,UDH] = COMPUTEOSCDIRICHLETL2(COORDINATES,ELEMENTS,UD) discretizes 
%   the given Dirichlet data by use of the L2 projection onto continuous 
%   affine splines and computes the corresponding Dirichlet data 
%   oscillations.
%
%   For a given partition {E1,...,EN} of the Dirichlet boundary GammaD, the
%   function returns the (N x 1)-vector OSC, where
%          
%      OSC(j) = diam(Ej) || (uD - uDh)' ||_{L2(Ej)}^2
%          
%   With the L2 orthogonal projection Pih onto continuous and affine 
%   splines, the discrete Dirichlet data read
%
%      uDh = Pih*uD
%
%   To compute the L2-projection, we use a Gaussian quadrature with n=3 
%   nodes which is exact for polynomials of degree 5 to integrate uD. With 
%   {z1,...,zn} the nodes on the Dirichlet boundary, the function 
%   additionally returns the (n x 1)-vector of nodal values of uDh, i.e., 
%   uDh(zj) = UDH(j).
%
%   For the implementation of the L2-norm, we transform the integration 
%   domain Ej onto the reference element [-1,1] and replace uD - Pih*uD by 
%   some polynomial p(t) which interpolates the integrand at the Gaussian 
%   quadrature nodes xk. Then, the integral int_{-1}^{+1} |p'(t)|^2 dt is 
%   again computed by the same quadrature rule, where the p'(xk) are 
%   computed with the help of the values p(xk).
%
%   COORDINATES gives the coordinates for (at least) all nodes on the 
%   Dirichlet boundary GammaD. ELEMENTS describes (precisely) the elements 
%   on GammaD. UD is a function handle for the Dirichlet data uD. For an 
%   (M x 2)-matrix of nodes X, it returns the (M x 1)-vector Y of 
%   corresponding function values, i.e. Y(j) := uD(X(j,:)).

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

%*** gauss(3) quadrature
quad_nodes = sqrt(3/5)*[-1;0;1];
quad_weights = [5,8,5]/9;

%*** general constants
nC = size(coordinates,1);
nE = size(elements,1);
nQ = size(quad_nodes,1);

%*** compute mesh-size
a = coordinates(elements(:,1),:);
b = coordinates(elements(:,2),:);
h = sqrt(sum((a-b).^2,2)); 

%*** build L2 mass matrix with respect to S1
I = elements(:,[1 1 2 2]);
J = elements(:,[1 2 1 2]);
M = sparse(I,J,h*[2 1 1 2]/6);

%*** perform all necessary evaluations of Dirichlet data uD
sx = reshape(a,2*nE,1)*(1-quad_nodes') + reshape(b,2*nE,1)*(1+quad_nodes');
sx = 0.5 * reshape(sx',nQ*nE,2);
uD_sx = reshape(uD(sx,varargin{:}),nQ,nE);

%*** build right-hand side vector for L2 projection onto S1
c = 0.25 * [h;h] .* [ quad_weights*(uD_sx.*repmat(1-quad_nodes,1,nE)), ...
                      quad_weights*(uD_sx.*repmat(1+quad_nodes,1,nE)) ]';
c = accumarray(elements(:),c);

%*** compute (nodal vector of) L2 projection uDh of uD onto S1
uDh = M\c;

%*** evaluate uDh at quadrature nodes
uDh_sx = 0.5 * ( (1-quad_nodes)*uDh(elements(:,1))' ...
                + (1+quad_nodes)*uDh(elements(:,2))' );

%*** compute oscillations of Dirichlet data via gauss(3) formula
D = 0.5 * sqrt(5/3) * [-3 -4 -1 ; -1 0 1 ; 1 4 3];
osc = 2*(quad_weights * ( D*(uD_sx - uDh_sx) ).^2)';

