function [osc,uDh] = computeOscDirichlet(coordinates,elements,uD)
%COMPUTEOSCDIRICHLET   Computes oscillations and discrete Dirichlet data.
%   [OSC,UDH] = COMPUTEOSCDIRICHLET(COORDINATES,ELEMENTS,UD) discretizes the
%   given Dirichlet data by nodal interpolation and computes the 
%   corresponding Dirichlet data oscillations.
%
%   For a given partition {E1,...,EN} of the Dirichlet boundary GammaD, the
%   function returns the (N x 1)-vector OSC, where
%          
%      OSC(j) = diam(Ej) || (uD - uDh)' ||_{L2(Ej)}^2
%          
%   With the nodal interpolation operator Ih onto continuous and affine 
%   splines, the discrete Dirichlet data read
%
%      uDh = Ih*uD
%
%   The L2-norm is computed via a closed Newton-Cotes rule with n=3 nodes,
%   which is exact for polynomials of degree 3. With {z1,...,zn} the nodes 
%   on the Dirichlet boundary, the function additionally returns the 
%   (N x 1)-vector of nodal values of uD, i.e., uD(zj) = uDh(zj) = UDH(j).
%
%   For the implementation of the L2-norm, we transform the integration 
%   domain Ej onto the reference element [-1,1] and replace uD - Ih*uD by 
%   some polynomial p(t) which interpolates at the nodes -1, 0, and +1. 
%   Then, the integral int_{-1}^{+1} |p'(t)|^2 dt is done via Newton-Cotes
%   quadrature, where p'(-1), p'(0), p'(+1) are computed with the help of 
%   the function values p(-1) = 0 = p(+1) and p(0). 
%
%   COORDINATES gives the coordinates for (at least) all nodes on the 
%   Dirichlet boundary GammaD. ELEMENTS describes (precisely) the elements 
%   on GammaD. UD is a function handle for the Dirichlet data uD. For an 
%   (M x 2)-matrix of nodes X, it returns the (M x 1)-vector Y of 
%   corresponding function values, i.e. Y(j) := uD(X(j,:)).

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

%*** compute midpoints of all elements
midpoints = 0.5*( coordinates(elements(:,1),:) ...
                + coordinates(elements(:,2),:) );

%*** evaluate Dirichlet data at element midpoints
uD_midpoints = uD(midpoints);

%*** evaluate Dirichlet data at all Dirichlet nodes
uDh = zeros(size(coordinates(:,1),1),1);
idx = unique(elements);
uDh(idx) = uD(coordinates(idx,:));

%*** compute oscillations of Dirichlet data via adapted Newton-Cotes formula
osc = 4/3*( uDh(elements(:,1))+uDh(elements(:,2))-2*uD_midpoints ).^2;

