function ind = computeEstSlpResidual(varargin)
%COMPUTEESTSLPRESIDUAL   Weighted-residual estimator for Symm's IE.
%   COMPUTEESTSLPRESIDUAL returns the local contributions of the weighted-
%   residual error estimator for a weakly-singular integral equation.
%
%   Usage: IND = COMPUTEESTSLPRESIDUAL([COORDINATES,ELEMENTS,PHIH,GH])
%      or  IND = COMPUTEESTSLPRESIDUAL([COORDINATES,ELEMENTS,PHIH,[],GH])
%      or  IND = COMPUTEESTSLPRESIDUAL([VERTICES,TRIANGLES,ELEMENTS, ...
%                                                               PHIH,GH,FH])
%      or  IND = COMPUTEESTSLPRESIDUAL([VERTICES,TRIANGLES,ELEMENTS, ...
%                                                               PHIH,[],FH])
%
%   This function returns the local contribution of the weighted-residual 
%   error estimator for the weakly singular integral equation
%
%      (1)  V*phi = g
%
%   and covers 4 different cases, based on the right-hand side g. For a 
%   coarse partition {E1,...,EN} of the boundary Gamma, the function returns
%   an (N x 1)-vector IND which contains the element-wise error 
%   contributions.
%
%   (A) Weakly singular integral equation
%
%   For the solution of (1) for general g, the function returns an 
%   (N x 1)-vector IND with
%
%      IND(j) = diam(Ej) || [gh - V*phih]' ||_{L2(Ej)}^2
%
%   where gh are the discretized data. In this case, the function needs to 
%   be called with parameters
%
%      [COORDINATES,ELEMENTS,PHIH,[],GH].
%
%   (B) Dirichlet problem
%
%   Here, g in (1) is of the form g = (K+1/2)*uD, where uD is the Dirichlet 
%   data. Then, (1) is equivalent to
%
%      (2)  -Laplace(u) = 0 in Omega
%                    u = uD on Gamma = boundary(Omega).
%
%   In this case, the function returns an (N x 1)-vector IND with
%
%      IND(j) = diam(Ej) || [rhs - V*phih]' ||_{L2(Ej)}^2
%
%   where rhs = (K+1/2)*gh with gh the discretized Dirichlet data. The 
%   parameters needed to call the function in this case are
%
%      [COORDINATES,ELEMENTS,PHIH,GH].
%
%   (C) Dirichlet problem with volume force
%
%   Here, g in (1) is of the form g = (K+1/2)*uD - N*f. Then, (1) is 
%   equivalent to
%
%      (3)  -Laplace(u) = f in Omega
%                     u = uD on Gamma = boundary(Omega).
%
%   In this case, the function returns an (N x 1)-vector IND with
%
%      IND(j) = diam(Ej) || [rhs - V*phih]' ||_{L2(Ej)}^2
%
%   where rhs = (K+1/2)*uDh - N*fh. The parameters needed to call the 
%   function in this case are
%
%      [VERTICES,TRIANGLES,ELEMENTS,PHIH,GH,FH].
%
%   Here, VERTICES and TRIANGLES describe the volume mesh. If GH = 0, the 
%   call of
%
%      [VERTICES,TRIANGLES,ELEMENTS,PHIH,[],FH]
%
%   avoids the call of evaluateK.
%
%   (D) Auxiliary problem for hypersingular equation with volume
%
%   Here, g in (1) is of the form g = N*f. In this case, the function 
%   returns an (N x 1)-vector IND with
%              
%      IND(j) = diam(Ej) || [rhs - V*phih|' ||_{L2(Ej)}^2
%
%   where rhs = N*fh. The parameters needed to call the function in this 
%   case are
%
%      [VERTICES,TRIANGLES,ELEMENTS,PHIH,[],-FH].
%
%   Here, VERTICES and TRIANGLES describe the volume mesh.
%
%   The function phih is the computed Galerkin solution, gh are the 
%   discretized (Dirichlet) data, and fh are the discretized Volume data. 
%
%   For the implementation, we transform the integration domain Ej onto the
%   reference element [-1,1] and replace the integrand by some polynomial 
%   p(t) which interpolates at three nodes. Then, the integral 
%   int_{-1}^{+1}|p'(t)|^2 dt is computed via the Gaussian quadrature rule 
%   with two nodes. Besides the two Gauss nodes, we use the midpoint 0 for 
%   the interpolation.
%
%   The matrices COORDINATES and ELEMENTS describe the coarse partition of 
%   Gamma. The matrices VERTICES and TRIANGLES describe the coarse partition
%   of Omega. The matrices DIRICHLET and NEUMANN describe the partition of 
%   the Dirichlet resp. Neumann boundary. The j-th entry PHIH(j) of column 
%   vector PHIH returns the value of phih on Ej. The k-th entry GH(k) 
%   provides the nodal value of uDh(zk), where zk is the k-th node of the 
%   given partition. The k-th entry FH(k) provides the value of fh on Tk
%   where Tk is the k-th element of the given volume partition.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

if nargin == 6
  [vertices,volumes,elements,phih,gh,fh] = varargin{:};
  coordinates = vertices(unique(elements),:);
elseif nargin == 5
  [coordinates,elements,phih,gh_dir,gh] = varargin{:};
elseif nargin == 4
  [coordinates,elements,phih,gh] = varargin{:};
end

%*** Gaussian quadrature on [-1,1] with 2 nodes and exactness 3
quad_nodes = [-1 1]/sqrt(3);
quad_weights = [1;1];

%*** elementwise interpolation is done in (gauss_left,gauss_right,midpoint)
quad_nodes(3) = 0;
nE = size(elements,1);
nQ = length(quad_nodes);

%*** build vector of evaluations points as (nQ*nE x 2)-matrix
a = coordinates(elements(:,1),:);
b = coordinates(elements(:,2),:);
sx = reshape(a,2*nE,1)*(1-quad_nodes) + reshape(b,2*nE,1)*(1+quad_nodes);
sx = 0.5*reshape(sx',nQ*nE,2);

%*** evaluate gh elementwise at (left, right, midpoint)
if ~isempty(gh)
  gh_left = gh(elements(:,1));
  gh_right = gh(elements(:,2));
  gh_sx = gh_left*(1-quad_nodes) + gh_right*(1+quad_nodes);
  gh_sx = 0.5*reshape(gh_sx',nQ*nE,1);
end

%*** evaluate V*phih in all interpolation nodes sx
p = evaluateV(coordinates,elements,phih,sx);

%*** distinguish between different cases
if (nargin == 6)
  if ~isempty(gh)
    p = p - evaluateK(coordinates,elements,gh,sx) ...
        - 0.5*gh_sx ...
        + evaluateN(vertices,volumes,fh,sx);
  else
    p = p + evaluateN(vertices,volumes,fh,sx);
  end
elseif (nargin == 5) && isempty(gh_dir)
  p = p - gh_sx;
elseif nargin == 4
  p = p - evaluateK(coordinates,elements,gh,sx) ...
      - 0.5*gh_sx;
end

%*** evaluate arclength-derivative p' elementwise at (left,right)
p_prime = reshape(p,nQ,nE)' * [-3 1 ; -1 3 ; 4 -4]*sqrt(0.75);

%*** return ind(j) = diam(Ej) * || [ V*phi - (K+1/2)*gh ]' ||_{L2(Ej)}^2
ind = 2*p_prime.^2*quad_weights;

