function ind = computeEstMixTau(father2dirichlet, father2neumann, ...
    neumann_coarse,neumann_fine,V_fine,K_fine,W_fine,bD_fine,bN_fine, ...
    x_coarse,free_neumann,free_neumann_fine)
%COMPUTEESTMIXTAU   (h-h/2)-based two-level estimator for mixed problem.
%   IND = COMPUTEESTMIXTAU(FATHER2DIRICHLET,FATHER2NEUMANN, ...
%   NEUMANN_COARSE,NEUMANN_FINE,V_FINE,K_FINE,W_FINE,BD_FINE,BN_FINE, ...
%   X_COARSE,FREE_NEUMANN,FREE_NEUMANN_FINE) computes the (h-h/2)-based 
%   two-level error estimator for the mixed problem.
%
%   We consider the mixed boundary value problem
%
%      -Laplace(u)  = f    in Omega
%               u   = uD   on GammaD
%             du/dn = phiN on GammaN
%
%   for given data uD, phiN, and f. Let {E1,...,EN} be a coarse partition of
%   the boundary Gamma. Let U_coarse = (u_coarse,phi_coarse) be the 
%   corresponding Galerkin solution. Let U_fine = (u_fine,phi_fine) be the
%   Galerkin solution with respect to the uniformly refined mesh. This 
%   function returns an (N x 1)-vector IND with
% 
%      IND(j) = ||| G_Ej (U_fine - U_coarse) |||^2
%
%   where G_Ej denotes the Galerkin projection onto span{(v_Ej,psi_Ej)}. The
%   two-level basis-function psi_Ej is just the fine-mesh basis function 
%   associated with the midpoint of the coarse-mesh element Ej whereas 
%   v_Ej := chi_Ej1-chi_Ej2, where chi_Eji denotes the characteristic 
%   function with respect to the two sons of Ej.
%
%   FATHER2DIRICHLET and FATHER2NEUMANN link the fine mesh with the coarse 
%   mesh. NEUMANN_COARSE and NEUMANN_FINE discribe the neumann boundary 
%   explicitly. The operator matrices V_FINE, K_FINE, W_FINE must be 
%   provided in the reduced forms (shrinked to the degrees of freedom) on 
%   the fine mesh. The solution vector X_COARSE is also given in the reduced
%   form
%
%      X=(DoF on GammaD, DoF on GammaN)
%
%   where the number of DoF on GammaD is the length of free_neumann. The 
%   right-hand side is given in the reduced form as the vectors BD_FINE 
%   (RHS associated with GammaD) and BN_FINE (RHS associated with GammaN).

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

nD_coarse = size(father2dirichlet,1);
dof_fine = 2*nD_coarse+size(K_fine,2);
Ndof_fine = length(free_neumann_fine);
Ndof = length(free_neumann);

%*** build index vector son2father to link fine mesh with coarse mesh
dirichlet2father = zeros(2*nD_coarse,1);
dirichlet2father(father2dirichlet) = repmat((1:nD_coarse)',1,2);

%*** build index field k = idx(j) such that j-th node of coarse mesh 
%*** coincides with k-th node of fine mesh
idx = zeros(2,1);
idx(neumann_coarse) = [ neumann_fine(father2neumann(:,1),1),...
                         neumann_fine(father2neumann(:,2),2) ];
idx=idx(free_neumann);

%*** build index field k = mid(j) such that midpoint of j-th element of 
%*** coarse mesh is k-th node of fine mesh
mid = neumann_fine(father2neumann(:,1),2);

%*** compute coefficient vector of u_coarse w.r.t. fine mesh
x = zeros(dof_fine,1);
x_coarse_tmp=zeros(2,1);
x_coarse_tmp(free_neumann)=x_coarse(1:Ndof);
x(mid) = 0.5*sum(x_coarse_tmp(neumann_coarse),2);
x(idx) = x_coarse(1:Ndof);
x=x(free_neumann_fine);
x(Ndof_fine+(1:2*nD_coarse))=x_coarse(Ndof+dirichlet2father);

%*** map dirichlet coordinates onto degrees of freedom on the dirichlet
%*** boundary
idxFN=zeros(2,1);
idxFN(free_neumann_fine)=1:Ndof_fine;

%*** compute residual w.r.t. the coarse mesh solution
A=[W_fine,K_fine';-K_fine,V_fine];
res = A*x-[bN_fine;bD_fine];

%*** compute engergy
energy = [diag(W_fine);V_fine(father2dirichlet(:,1)...
    + 2*nD_coarse*(father2dirichlet(:,1) - 1))...
    - V_fine(father2dirichlet(:,1) ...
    + 2*nD_coarse*(father2dirichlet(:,2) - 1))];

%*** compute estimator
ind=[res(idxFN(mid)).^2./energy(idxFN(mid));...
    (res(father2dirichlet(:,1)+Ndof_fine)...
    -res(father2dirichlet(:,2)+Ndof_fine)).^2 ...
    ./(2*energy(Ndof_fine+1:end))];

