function [indSLP,indHYP] = computeEstMixResidual(varargin)
%COMPUTEESTMIXRESIDUAL   Weighted-residual estimator for mixed problem.
%   COMPUTEESTMIXRESIDUAL computes the weighted-residual based error 
%   estimator for the mixed problem.
%
%   USAGE: 
%    (1) [INDSLP,INDHYP] = COMPUTEESTMIXRESIDUAL(COORDINATES,DIRICHLET, ...
%                                               NEUMANN,UNH,PHIDH,UDH,PHINH)
%   or
%
%    (2) [INDSLP,INDHYP] = COMPUTEESTMIXRESIDUAL(VERTICES,COORDINATES, ... 
%                  VOLUMES,DIRICHLET,NEUMANN,UNH,PHIDH,UDH,PHINH,FH,LAMBDAH)
%               
%   We consider the mixed boundary value problem
%
%      -Laplace(u)  = f    in Omega
%               u   = uD   on GammaD
%             du/dn = phiN on GammaN
%
%   for given data uD, phiN, and f. Let {E1,...,EN} be a coarse partition of
%   the boundary Gamma. Let U_coarse = (u_coarse,phi_coarse) be the 
%   corresponding Galerkin solution. Let U_fine = (u_fine,phi_fine) be the
%   Galerkin solution with respect to the uniformly refined mesh. This 
%   function returns an (N x 1)-vector IND with
% 
%      IND(j) = diam(E_j)[||(res_D)'||^2_L2(E_j)+ ||res_N||^2_L2(E_j)]
%
%   where
%			
%      res_D := V PhiDh - (1/2+K) UDh + V PhiNh - K UNh
%      res_N := (1/2-K') PhiNh - K'PhiDh - W(UNh+ UDh).
%
%   The arrays COORDINATES DIRICHLET, NEUMANN, and VERTICES, VOLUMES 
%   describe the boundary mesh as well as the volume mesh. The vectors UNh 
%   and PhiDh describe the solution of the integral equations. Here, the 
%   number of entries of UNh is equal to the number of coordinates, but UNh
%   is non-zero only at indices which correspond to degrees of freedom on 
%   the Neumann boundary. The values in PhiDh correspond to the elementwise
%   values of Phi_ell on the Dirichlet boundary. The discretized data are 
%   given as the vectors UDh and PhiNh.  
%		
%   In case of non-homogeneous volume forces, the discretized volume force 
%   Fh as well as the Galerkin approximation Lambdah of N_0Fh has to be 
%   provided.

% (C) 2009-2012 - 2012 HILBERT-Team '12
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

if nargin == 11
  [vertices,coordinates,volumes,dirichlet,neumann,uNh,...
        phiDh,uDh,phiNh,fh,lambdah] = varargin{:};
  elements=[dirichlet;neumann];
elseif nargin == 7
  [coordinates,dirichlet,neumann,uNh,phiDh,uDh,phiNh] = varargin{:};
  elements=[dirichlet;neumann];
end
%***compute first equation estimate
%*** Gaussian quadrature on [-1,1] with 2 nodes and exactness 3
quad_nodes = [-1 1]/sqrt(3);
quad_weights = [1;1];

%*** elementwise interpolation is done in (gauss_left,gauss_right,midpoint)
quad_nodes(3) = 0;
nE = size(elements,1);
nQ = length(quad_nodes);

%*** build vector of evaluations points as (nQ*nE x 2)-matrix
a = coordinates(elements(:,1),:);
b = coordinates(elements(:,2),:);
sx = reshape(a,2*nE,1)*(1-quad_nodes) + reshape(b,2*nE,1)*(1+quad_nodes);
sx = 0.5*reshape(sx',nQ*nE,2);

%*** evaluate gh elementwise at (left, right, midpoint)

uDh_left = uDh(elements(:,1));
uDh_right = uDh(elements(:,2));
uDh_sx = uDh_left*(1-quad_nodes) + uDh_right*(1+quad_nodes);
uDh_sx = 0.5*reshape(uDh_sx',nQ*nE,1);
pSLP = evaluateV(coordinates,dirichlet,phiDh,sx) ...
          - evaluateK(coordinates,elements,uDh,sx) ...
          - 0.5*uDh_sx ...
          + evaluateV(coordinates, elements, phiNh,sx) ...
          - evaluateK(coordinates,neumann,uNh,sx);
if nargin == 11
    pSLP = pSLP + evaluateN(vertices, volumes,fh,sx);
end

%*** evaluate arclength-derivative p' elementwise at (left,right)
pSLP_prime = reshape(pSLP,nQ,nE)' * [-3 1 ; -1 3 ; 4 -4]*sqrt(0.75);

%*** return ind(j) = diam(Ej) * || [ V*phi - (K+1/2)*gh ]' ||_{L2(Ej)}^2
indSLP = 2*pSLP_prime.^2*quad_weights;

%***compute second equation estimate
quad_nodes(3)=[];
nQ=length(quad_nodes);
%*** build vector of all quadrature nodes as (nQ*nE x 2)-matrix
a = coordinates(elements(:,1),:);
b = coordinates(elements(:,2),:);
sx = reshape(a,2*nE,1)*(1-quad_nodes) ...
        + reshape(b,2*nE,1)*(1+quad_nodes);
sx = 0.5*reshape(sx',nQ*nE,2);

%*** sx2element(j) returns the element number k such that sx(j,:) lies on Ek
sx2element = reshape(repmat((1:nE),nQ,1),nQ*nE,1);
%*** compute vector of (squared) element-widths
h = sum((a-b).^2,2); 

%*** compute outer normal vector
normal = b-a;
normal = [normal(:,2),-normal(:,1)]./repmat(sqrt(h),1,2);

%*** evaluate p1 = (1/2-Kadj)*phih at (left,right)
pHYP = 0.5*phiNh(sx2element) ...
   - evaluateKadj(coordinates,elements,phiNh,sx,normal(sx2element,:))...
   - evaluateKadj(coordinates,dirichlet,phiDh,sx,normal(sx2element,:));

if nargin == 11
   pHYP=pHYP - evaluateKadj(coordinates,elements,lambdah,sx, ... 
       normal(sx2element,:)) +0.5*lambdah(sx2element);
end

pHYP=pHYP-evaluateW(coordinates,elements,uDh+uNh,sx,normal(sx2element,:));

%*** evaluate integrand p = (1/2-Kadj)*phih - W*gh = p1 + p2'
pHYP = reshape(pHYP,nQ,nE)';

%*** return ind(j) = diam(Ej) * ||(1/2-K*)phih+(Vgh')'||_{L2(Ej)}^2
indHYP = 0.5*h.*(pHYP.^2*quad_weights);
