function ind = computeEstHypResidual(varargin)
%COMPUTEESTHYPRESIDUAL   Weighted-residual estimator for hypsing IE.
%   COMPUTEESTHYPRESIDUAL returns the local contributions of the weighted-
%   residual error estimator for a hyper-singular integral equation.
%
%   Usage: IND = COMPUTEESTHYPRESIDUAL([COORDINATES,ELEMENTS,GH,PHIH])
%      or  IND = COMPUTEESTHYPRESIDUAL([COORDINATES,ELEMENTS,GH,[],PHIH])
%      or  IND = COMPUTEESTHYPRESIDUAL([COORDINATES,ELEMENTS,GH,PHIH, ...
%                                                               LAMBDAH])
%
%   This function returns the local contribution of the weighted-residual 
%   error estimator for the hypersingular integral equation
%
%      (1) W*g = phi
%
%   and covers 3 different cases, based on the right-hand side phi. For a 
%   coarse partition {E1,...,EN} of the boundary Gamma, the function returns
%   an (N x 1)-vector IND which contains the element-wise error 
%   contributions.
%
%   (A) Hypersingular integral equation
%
%   For a Galerkin-solution gh of (1) for general phi, the function returns
%   an (N x 1)-vector IND with
%
%      IND(j) = diam(Ej) || phih - W*gh ||_{L2(Ej)}^2
%
%   where phih are the discretized data. In this case, the function needs 
%   to be called with parameters
%
%      [COORDINATES,ELEMENTS,GH,[],PHIH].
%
%   (B) Neumann problem
%
%   Here, phi in (1) is of the form phi = (-1/2 + K')uN, where uN is the 
%   Neumann data. Then, (1) is equivalent to
%
%      (2)  -Laplace(u) = 0 in Omega
%          \partial_n u = uN on Gamma = boundary(Omega).
%
%   In this case, the function returns an (N x 1)-vector IND with
%
%      IND(j) = diam(Ej) || rhs - W*gh ||_{L2(Ej)}^2
%
%   where rhs = (-1/2 + K')phih with phih the discretized Neumann data. The
%   parameters needed to call the function in this case are
%
%      [COORDINATES,ELEMENTS,GH,PHIH].
%
%   (C) Neumann problem with volume force
%
%   Here, phi in (1) is of the form phi = (-1/2+K')uN - N_1f. Then, (1) is 
%   equivalent to
%
%      (3)  -Laplace(u) = f in Omega
%          \partial_n u = uN on Gamma = boundary(Omega).
%
%   In this case, the function returns an (N x 1)-vector IND with
%
%      IND(j) = diam(Ej) || rhs - W*gh ||_{L2(Ej)}^2
%
%   where rhs = (-1/2+K')(phih + lambdah), where phih is the discretized 
%   Neumann data and lambdah is the Galerkin solution of the weakly-singular
%   integral equation
%
%      V*lambdah = N*fh,
%
%   where fh is the discretized volume force.The parameters needed to call 
%   the function in this case are
%
%      [COORDINATES,ELEMENTS,GH,PHIH,LAMBDAH].
%
%   For the implementation, we use a 2-point Gauss rule which is exact for 
%   polynomials of degree 3.
%
%   The matrices COORDINATES and ELEMENTS describe the partition of Gamma. 
%   The matrices VERTICES and TRIANGLES describe the partition of OMEGA. The
%   matrices DIRICHLET and NEUMANN describe the partition of the Dirichlet 
%   resp. Neumann boundary. The k-th entry GH(k) provides the nodal value of
%   gh(zk), where zk is the k-th node of the given partition. The j-th entry
%   PHIH(j) of the column vector PHIH returns the value of phih on element 
%   Ej.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1
if nargin == 5
  [coordinates,elements,gh,phih_dir,phih_ind] = varargin{:};
elseif nargin == 4
  [coordinates,elements,gh,phih_dir] = varargin{:};
end

%*** Gaussian quadrature on [-1,1] with 2 nodes and exactness 3
quad_nodes = [-1 1]/sqrt(3);
quad_weights = [1;1];

%*** define constants
nE = size(elements,1);
nQ = length(quad_nodes);

%*** build vector of all quadrature nodes as (nQ*nE x 2)-matrix
a = coordinates(elements(:,1),:);
b = coordinates(elements(:,2),:);
sx = reshape(a,2*nE,1)*(1-quad_nodes) ...
        + reshape(b,2*nE,1)*(1+quad_nodes);
sx = 0.5*reshape(sx',nQ*nE,2);

%*** sx2element(j) returns the element number k such that sx(j,:) lies on Ek
sx2element = reshape(repmat((1:nE),nQ,1),nQ*nE,1);

%*** compute vector of (squared) element-widths
h = sum((a-b).^2,2);

%*** compute outer normal vector
normal = b-a;
normal = [normal(:,2),-normal(:,1)]./repmat(sqrt(h),1,2);

%*** evaluate the the associated terms
if (nargin == 5) && isempty(phih_dir)
  p = evaluateW(coordinates,elements,gh,sx,normal(sx2element,:)) ...
    - phih_ind(sx2element);
else
  p = evaluateW(coordinates,elements,gh,sx,normal(sx2element,:)) ...
    - 0.5*phih_dir(sx2element) ...
    + evaluateKadj(coordinates,elements,phih_dir,sx,normal(sx2element,:));
  if nargin == 5
    p = p + evaluateKadj(coordinates,elements,phih_ind,sx, ...
        normal(sx2element,:)) - 0.5*phih_ind(sx2element);
  end
end

p = reshape(p,nQ,nE)';
ind = 0.5*h.*((p.^2)*quad_weights);

