function err = computeErrNeumann(coordinates,elements,p,phi)
%COMPUTEERRNEUMANN   Error between a function and P0-approximation.
%   ERR = COMPUTEERRNEUMANN(COORDINATES,ELEMENTS,P,PHI) computes the error 
%   between a given function phi and some P0-approximation p in a norm 
%   associated with the Neumann data.
%
%   For a given partition {E1,...,EN} of the Neumann boundary GammaN, the 
%   function returns the (N x 1)-vector ERR, where
%
%      ERR(j) = diam(Ej) * || phi - phih ||_{L2(Ej)}^2
%              
%   Here, phih denotes a piecewise constant approximation of phi. The 
%   L2-norm is computed via a Gaussian quadrature rule with n=2 nodes, i.e.
%   exactness 2*n-1 = 3.
%
%   COORDINATES gives the coordinates of (at least) all nodes on GammaN. 
%   ELEMENTS gives (precisely) the elements on GammaN. P is an 
%   (N x 1)-vector containing the elementwise values of phih. PHI is a 
%   function handle for the Neumann data. For an (M x 2)-matrix of nodes X 
%   and the respective boundary elements given by the (M x 2) matrices A 
%   and B, it returns the (M x 1)-vector Y of corresponding function values,
%   i.e. Y(j) := PHI(X(j,:)).

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

%*** arbitrary quadrature on [-1,1] with exactness n >= 2, e.g., gauss(2)
quad_nodes = [-1 1]/sqrt(3);
quad_weights = [1;1];

%*** the remaining code is independent of the chosen quadrature rule
nE = size(elements,1);
nQ = length(quad_nodes);

%*** build vector of evaluations points as (nQ*nE x 2)-matrix
a = coordinates(elements(:,1),:);
b = coordinates(elements(:,2),:);
sx = reshape(a,2*nE,1)*(1-quad_nodes) + reshape(b,2*nE,1)*(1+quad_nodes);
sx = 0.5*reshape(sx',nQ*nE,2);

%*** phi(sx) usually depends on the normal vector, whence phi takes sx and 
%*** the nodes of the respective element to compute the normal
a_sx = reshape(repmat(reshape(a,2*nE,1),1,nQ)',nE*nQ,2);
b_sx = reshape(repmat(reshape(b,2*nE,1),1,nQ)',nE*nQ,2);

%*** perform all necessary evaluations of phi as (nE x nQ)-matrix
phi_sx = reshape(phi(sx,a_sx,b_sx),nQ,nE)';

%*** compute vector of (squared) element-widths
h = sum((a-b).^2,2); 

%*** compute Neumann error simultaneously for all elements
err_sx = (phi_sx - repmat(reshape(p,nE,1),1,nQ)).^2;
err = 0.5*h.*(err_sx*quad_weights);

