function err = computeErrDirichlet(coordinates,elements,g,uD)
%COMPUTEERRDIRICHLET   Error between a function and S1-approximation.
%   ERR = COMPUTEERRDIRICHLET(COORDINATES,ELEMENTS,G,UD) computes the error
%   between a given function uD and some S1-approximation g in a seminorm 
%   associated with the Dirichlet data.
%
%   For a given partition {E1,...,EN} of a boundary piece Gamma, the 
%   function returns the (N x 1)-vector ERR, where
%          
%      ERR(j) = diam(Ej) * || (uD - uDh)' ||_{L2(Ej)}^2
%          
%   Here, uDh denotes a continuous and piecewise affine approximation of the
%   Dirichlet data uD. The L2-norm is computed via a closed Newton-Cotes 
%   rule with n=3 nodes, which is exact for polynomials of degree 3. 
%
%   For the implementation, we transform the integration domain Ej onto the
%   reference element [-1,1] and replace uD - uDh by some polynomial p(t) 
%   which interpolates at the nodes -1, 0, and +1. Then, the integral 
%   int_{-1}^{+1} |p'(t)|^2 dt is done via Newton-Cotes quadrature, where 
%   p'(-1), p'(0), p'(+1) are computed with the help of the function values
%   p(-1), p(+1) and p(0).
%
%   COORDINATES gives the coordinates of (at least) all nodes on Gamma. 
%   ELEMENTS gives (precisely) the elements on Gamma. G is the vector of 
%   nodal values of uDh. UD is a function handle for the Dirichlet data uD.
%   For an (M x 2)-matrix of nodes X, it returns the (M x 1)-vector Y of 
%   corresponding function values, i.e. Y(j) := uD(X(j,:)).

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

%*** compute midpoints of all elements
midpoints = 0.5*( coordinates(elements(:,1),:) ...
                                    + coordinates(elements(:,2),:) );

%*** compute p = (uD - uDh) at element midpoints
p_midpoints = uD(midpoints) - 0.5*sum(g(elements),2);

%*** compute p = (uD - uDh) at all nodes
p_nodes = uD(coordinates) - g;

%*** evaluate derivative p' at all elements (left,midpoint,right)
p_prime = [p_nodes(elements) p_midpoints] * [-3 -1 1 ; -1 1 3 ; 4 0 -4]*0.5;
      
%*** compute Dirichlet error simultaneously for all elements
err = 2*p_prime.^2*[1;4;1]/3;

