function [varargout] = buildSortedMesh(varargin)
%BUILDSORTEDMESH   Provides a numbering of the nodes.
%   BUILDSORTEDMESH provides a numbering of the nodes such that certain 
%   nodes are first. Moreover, it can be used to create the induced 
%   boundary mesh from a given volume mesh.
%
%   Usage:  [VERTICES,TRIANGLES, [COORDINATES,] ELEMENTS] ...
%              = BUILDSORTEDMESH(VERTICES,TRIANGLES)
%  
%      or   [VERTICES,TRIANGLES, [COORDINATES,] <BOUNDARIES>] ...
%              = BUILDSORTEDMESH(VERTICES,TRIANGLES,<BOUNDARIES>]
%
%      or   [COORDINATES,<BOUNDARIES>] ...
%              = BUILDSORTEDMESH(COORDINATES,<BOUNDARIES>)
%
%   where BOUNDARIES is a synonym for DIRICHLET, NEUMANN, ROBIN, ...
%   so that any finite partition of the boundary Gamma into boundary parts
%   can be managed.
%
%   In some cases, our implementation of adaptive BEM uses a certain 
%   numbering {z1,...,zC} of the nodes of a boundary partition.  
%
%   CASE 1: When dealing with mixed boundary conditions, the boundary Gamma
%   is split into finitely many boundary pieces, e.g., in Dirichlet and 
%   Neumann boundary which are given in terms of (N x 2) resp. (D x 2) 
%   arrays NEUMANN and DIRICHLET. Our implementation uses that the nodes on
%   the Neumann boundary are the first n nodes contained in COORDINATES.
%   This is guaranteed by call of
%
%      [COORDINATES,NEUMANN,DIRICHLET] ...
%                = BUILDSORTEDMESH(COORDINATES,NEUMANN,DIRICHLET)
%
%   CASE 2: When dealing with Symm's integral equation or the hypersingular
%   integral with volume forces, we deal with coupled meshes only, i.e., 
%   the volume triangulation {T1,...,TM} is given and the boundary 
%   partition {E1,...,EN} is generated thereof. To build the integral 
%   operators K and W correctly (i.e. only for boundary nodes), it has to 
%   be guaranteed that the first nodes given by VERTICES belong to the 
%   boundary. This is guaranteed by call of
% 
%      [VERTICES,TRIANGLES, [COORDINATES,] ELEMENTS] ...
%                   = BUILDSORTEDMESH(VERTICES,TRIANGLES)
%
%   CASE 3: When dealing with mixed boundary conditions and volume forces,
%   the volume triangulation as well as the partition of the boundary Gamma
%   into Dirichlet and Neumann boundary is given. To build the integral 
%   operators K and W correctly (i.e. only for boundary nodes), it has to 
%   be guaranteed that the first nodes given by VERTICES belong the 
%   boundary. Moreover, the very first nodes have to belong to the Neumann 
%   boundary. This is guaranteed by call of
%
%      [VERTICES,TRIANGLES, [COORDINATES,] NEUMANN,DIRICHLET] ...
%                = BUILDSORTEDMESH(VERTICES,TRIANGLES,NEUMANN,DIRICHLET)
%
%   We stress that in CASE 1 and CASE 3, our implementation is able to deal
%   with any finite partition of Gamma into parts, e.g. Dirichlet, Neumann,
%   Robin boundary etc.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

%*** CASE 2 is covered by recursion to CASE 3, i.e., bdry mesh is generated
if nargin == 2
    %*** only volume mesh is given
    vertices = varargin{1};
    triangles = varargin{2};

    %*** create list of edges and sort node numbers per edge
    edges = [triangles(:,[1 2]); triangles(:,[2 3]); triangles(:,[3 1])];
    edges_sorted = sort(edges,2);

    %*** an edge is a boundary edge iff it appears once in edges_sorted
    [foo,sort2unique,unique2sort] = unique(edges_sorted,'rows');
    idx = sort2unique(accumarray(unique2sort,1) == 1);
    elements = edges(idx,:);

    %*** recursive call of buildSortedMesh, elements is now bdry part
    if nargout == 3
        %*** return value = [vertices,triangles,elements]
        [varargout{1},varargout{2},varargout{3}] ...
            = buildSortedMesh(vertices,triangles,elements);
    elseif nargout == 4
        %*** return value = [vertices,triangles,coordinates,elements]
        [varargout{1},varargout{2},varargout{3},varargout{4}] ...
            = buildSortedMesh(vertices,triangles,elements);
    end
    
    return;
end

%
%*** from now on, only CASE 1 and CASE 3 have to be considered
%

%*** second input parameter is volume mesh
if size(varargin{2},2) == 3
    input_offset = 3;
else
    input_offset = 2;
end

%*** user wants coordinates of bdry parts
if nargout == nargin + 1
    output_offset = 1;
else
    output_offset = 0;
end

%*** all nodes which belong to the volume and boundary mesh
nodes = varargin{1};
nN = size(nodes,1);

%*** build list of all boundary nodes after appearance
list_nodes = [];
for i = input_offset:nargin
    current_nodes = unique(varargin{i});
    list_nodes = [ list_nodes ; setdiff(current_nodes,list_nodes) ];
end

%*** number of boundary nodes
nC = size(list_nodes,1);

%*** add to list the remaining nodes which are not on the boundary
list_nodes = [list_nodes;setdiff(1:nN,list_nodes)'];

%*** build permutation such that vertices are sorted according to boundary
%*** parts
[foo,permutation] = sort(list_nodes);

%*** permute indices of nodes such that boundary nodes are first
nodes(permutation',:) = nodes;
varargout{1} = nodes;

%*** if coordinates of bdry nodes are wanted, return 
%*** varargout{3} = coordinates
if output_offset == 1
    varargout{3} = nodes(1:nC,:);
end

%*** if a volume mesh was given, return varargout{2} = triangles
if input_offset == 3
    varargout{2} = permutation(varargin{2}); 
end

%*** update and return boundary parts
for i = input_offset:nargin
    varargout{i + output_offset} = permutation(varargin{i});
end

