function [b_nodes,b_elements,lambdah] = buildMixedVolRHS(coordinates, ... 
                   dirichlet,neumann,uDh,phiNh,vertices,triangles,fh,V,K,W)
%BUILDMIXEDVOLRHS   Right-hand side for mixed problem with volume force.
%   [B_NODES,B_ELEMENTS,LAMBDAH] = BUILDMIXEDVOLRHS(COORDINATES, ... 
%                  DIRICHLET,NEUMANN,UDH,PHINH,VERTICES,TRIANGLES,FH,V,K,W)
%   assembles the right-hand side for an integral formulation of a mixed 
%   boundary value problem with non-homogeneous volume force with 
%   (S1 x P0)-ansatz and test functions.
%
%   The integral formulation of the mixed boundary value problem with 
%   non-vanishing volume force reads
%
%      A*(uN; phiD) = (1/2 - A)*(uD; phiN) - (N0*f; N1*f),
%
%   where A is the operator matrix (-K V ; W K'). This is an equivalent 
%   formulation of the PDE
%
%      -Laplace(u) = f    in Omega
%            du/dn = phiN on GammaN
%                u = uD   on GammaD
%
%   The normal derivative N1*f of the Newton Potential can be described 
%   equivalently by
%
%      N1 = (-1/2 + K')*V^(-1)*N
%
%   where N is the trace of the Newton potential on Gamma. This means that 
%   we first solve some indirect boundary integral equation
%
%      V*lambda = N0*f
%
%   In a second step, this is used to solve
%
%      A*(uN; phiD) = (1/2 - A)*(uD; phiN) - (N0*f; (1/2 - K')*lambda).
%
%   Let {E1,...,EN} be a partition of Gamma into affine line segments Ej. 
%   Let {T1,...,TM} be a regular triangulation of Omega into triangles Tj. 
%   For the numerical solution of the mixed boundary value problem with 
%   non-vanishing volume force by use of a Galerkin BEM with S1 x P0 Ansatz
%   and test functions, we replace uD by some piecewise affine and 
%   continuous spline uDh, phiN by some piecewise constant function phiNh, 
%   and the volume force f by some piecewise constant fh. On the continuous
%   level, the right hand side of the mixed BVP is then given by
%
%      (1/2 - A)*(uDh; phiNh) - (N0*fh; (1/2 - K')*lambdah),
%
%   where lambdah is the solution of
%
%      V*lambdah = N0*fh.
%
%   COORDINATES gives the coordinates for all nodes {z1,...,zN} on Gamma. 
%   Let {E1,...EN} describe the boundary elements. DIRICHLET and NEUMANN 
%   describe the elements on the Dirichlet boundary GammaD and the Neumann 
%   boundary GammaN. VERTICES describes the nodes of the volume 
%   triangulation TRIANGLES. The (N x 1)-column vector UDH provides the 
%   nodal values of the discrete Dirichlet data, i.e., uDh(zj) = UDH(j). 
%   The (N x 1)-column vector PHINH describes the elements-wise values of 
%   the discrete Neumann data, i.e., phiNh|_{Ej} = PHINH(j). The 
%   (M x 1)-column vector FH describes the elementwise values of fh, i.e., 
%   fh|_{Tj} = FH(j). The matrices V,K, and W are the discrete simple layer,
%   double layer and hypersingular operator.
%
%   The function returns the (N x 1)-column vectors B_NODES and B_ELEMENTS,
%   where
%      B_NODES(j) = rhs for j-th hat-function associated with node zj
%      B_ELEMENTS(j) = rhs for j-th characteristic function for Ej
%
%   Then, The right-hand side for the integral formulation of the mixed 
%   boundary value problem is built of appropriate subvectors of B_NODES 
%   and B_ELEMENTS.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

elements = [dirichlet;neumann];

%*** compute N-matrix for P0(Gamma) x P0(Omega)
N = buildN(coordinates,elements,vertices,triangles);

%*** compute N*fh and free unnecessary memory
Nfh = N*fh;
clear N;

%*** solve Symm's IE for the computation of N_1*f
lambdah = V\Nfh;

%*** compute mass-type matrix for P0 x S1
M = buildM(coordinates,elements);

%*** compute full right-hand side
b_elements = M*uDh*0.5 + K*uDh - V*phiNh-Nfh;
phiNh = phiNh + lambdah;
b_nodes = (0.5*phiNh'*M - phiNh'*K - uDh'*W)';

